<?php

declare(strict_types=1);

namespace Drupal\Tests\drupal_purview\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\drupal_purview\Service\PurviewAuthenticationService;
use Drupal\drupal_purview\Testing\FixtureLoaderTrait;
use Drupal\drupal_purview\Testing\HttpClientFactory;
use GuzzleHttp\Psr7\Response;

/**
 * Kernel tests for failure handling in the Purview API client.
 *
 * Verifies two scenarios:
 * - If authentication fails and no token is available, the client does not
 *   attempt an HTTP call and simply returns NULL.
 * - If Purview returns a 5xx error, the client gracefully returns NULL instead
 *   of throwing, allowing the caller to decide how to handle the failure.
 *
 * @group drupal_purview
 * @covers \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
 */
final class PurviewAuthAndHttpFailureKernelTest extends KernelTestBase {
  use FixtureLoaderTrait;

  /**
   * Minimal module set required to bootstrap the service container.
   *
   * @var string[]
   */
  protected static $modules = [
    'system',
    'user',
    'key',
    'drupal_purview',
  ];

  /**
   * Basic configuration applied before each test run.
   *
   * - Adds a fake Purview REST endpoint.
   * - Leaves authentication and HTTP client wiring up to individual tests.
   */
  protected function setUp(): void {
    parent::setUp();

    $this->config('drupal_purview.settings')
      ->set('rest_endpoint', 'https://example.purview')
      ->save();
  }

  /**
   * Ensures no HTTP request is made when authentication fails.
   *
   * Steps:
   * 1. Stub the authentication service to always return NULL.
   * 2. Register an "explosive" HTTP client that would 500 if called.
   * 3. Invoke the client and confirm it returns NULL without touching HTTP.
   */
  public function testNoHttpWhenAuthFails(): void {
    // Stub auth: no token available.
    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getAccessToken'])
      ->getMock();
    $auth->method('getAccessToken')->willReturn(NULL);
    $this->container->set('drupal_purview.auth_service', $auth);

    // "Explosive" HTTP client would throw if called.
    $bomb = HttpClientFactory::fromResponses([
      new Response(500, ['Content-Type' => 'application/json'], '{"error":"should-not-be-called"}'),
    ]);
    $this->container->set('http_client', $bomb);

    /** @var \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient $sut */
    $sut = $this->container->get('drupal_purview.api_client');

    $result = $sut->getManagedAttributeDefinitions(TRUE);
    $this->assertNull($result, 'Client returns NULL when no access token is available.');
  }

  /**
   * Ensures an HTTP 5xx error is handled gracefully.
   *
   * Steps:
   * 1. Stub the authentication service to provide a valid token.
   * 2. Register an HTTP client that always returns 500.
   * 3. Call the API and confirm the client swallows the error and returns NULL.
   */
  public function testHttpErrorHandledGracefully(): void {
    // Stub auth: valid token provided.
    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getAccessToken'])
      ->getMock();
    $auth->method('getAccessToken')->willReturn('test-token');
    $this->container->set('drupal_purview.auth_service', $auth);

    // HTTP client simulates a 500 error from Purview.
    $failing = HttpClientFactory::fromResponses([
      new Response(500, ['Content-Type' => 'application/json'], '{"error":"boom"}'),
    ]);
    $this->container->set('http_client', $failing);

    /** @var \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient $sut */
    $sut = $this->container->get('drupal_purview.api_client');

    $result = $sut->getManagedAttributeDefinitions(TRUE);
    $this->assertNull($result, 'Client returns NULL on server error instead of throwing.');
  }

}
