<?php

declare(strict_types=1);

namespace Drupal\Tests\drupal_purview\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\MemoryBackend;
use Drupal\Core\Cache\NullBackend;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\drupal_purview\Service\PurviewAuthenticationService;
use Drupal\drupal_purview\Service\PurviewClassicTypesApiClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Psr\Log\LoggerInterface;

/**
 * Behavior tests for PurviewClassicTypesApiClient.
 *
 * - Happy-path fetches for glossaries and term metadata
 * - Request shaping & paging for glossary term search
 * - Managed attribute typedef parsing (enum/array<enum>) + caching
 * - Grouping helper for managed attributes
 * - Failure paths: no token / RequestException => NULL.
 *
 * Uses Guzzle MockHandler + in-memory cache; no kernel boot required.
 *
 * @group drupal_purview
 * @covers \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
 */
final class PurviewClassicTypesApiClientTest extends UnitTestCase {

  // ---------- Tests ----------

  /**
   * Happy path: issues GET to /catalog/api/atlas/v2/glossary and returns array.
   */
  public function testGetGlossariesHappyPath(): void {
    $history = [];
    $payload = [['guid' => 'g1', 'name' => 'Glossary']];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($payload)),
    ], $history);

    $sut = $this->buildSut($http, $this->cacheBin(), 'token', 'https://example.purview/');
    $out = $sut->getGlossaries();

    $this->assertIsArray($out);
    $this->assertSame('g1', $out[0]['guid']);

    $req = $history[0]['request'];
    $this->assertSame('GET', $req->getMethod());
    $this->assertStringEndsWith('/catalog/api/atlas/v2/glossary', (string) $req->getUri());
  }

  /**
   * No token short-circuits and returns NULL without HTTP.
   */
  public function testGetGlossariesNoToken(): void {
    $history = [];
    $http = $this->httpWithHistory([], $history);

    $sut = $this->buildSut($http, $this->cacheBin(), NULL);
    $this->assertNull($sut->getGlossaries());
    $this->assertCount(0, $history);
  }

  /**
   * RequestException is caught: returns NULL and logs (no throw).
   */
  public function testGetGlossariesRequestException(): void {
    $history = [];
    $request = new Request(
      'GET',
      'https://example.purview/catalog/api/atlas/v2/glossary'
    );
    $http = $this->httpWithHistory([
      new Response(500, [], 'boom'),
    ], $history);

    // Force exception using http_errors=true, wrapping with RequestException.
    // Simplest: throw explicitly via MockHandler by using RequestException.
    $mock = new MockHandler([
      new RequestException('things broke', $request, new Response(500, [], 'boom')),
    ]);
    $http = new Client(['handler' => HandlerStack::create($mock), 'http_errors' => TRUE]);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->atLeastOnce())->method('error');

    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()->onlyMethods(['getAccessToken'])->getMock();
    $auth->method('getAccessToken')->willReturn('token');

    $sut = new PurviewClassicTypesApiClient(
      $this->configFactory(),
      $http,
      $this->loggerFactory($logger),
      $auth,
      $this->cacheBin(),
    );

    $this->assertNull($sut->getGlossaries());
  }

  /**
   * Shapes request (keywords/limit/offset/filter) and pages until short page.
   */
  public function testSearchGlossaryTermsPagingAndShaping(): void {
    $history = [];
    // First page returns 3 items (limit=3 in this test),
    // second page returns 1 -> stop.
    $page1 = ['value' => [['id' => 1], ['id' => 2], ['id' => 3]]];
    $page2 = ['value' => [['id' => 4]]];

    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($page1)),
      new Response(200, ['Content-Type' => 'application/json'], json_encode($page2)),
    ], $history);

    $sut = $this->buildSut($http);
    $out = $sut->searchGlossaryTerms('revenue', 3, 'Finance');

    $this->assertSame([['id' => 1], ['id' => 2], ['id' => 3], ['id' => 4]], $out);
    $this->assertCount(2, $history, 'Two POSTs due to paging');

    // Assert first request body.
    $b1 = json_decode((string) $history[0]['request']->getBody(), TRUE);
    $this->assertSame('revenue', $b1['keywords']);
    $this->assertSame(3, $b1['limit']);
    $this->assertSame(0, $b1['offset']);
    $this->assertArrayHasKey('filter', $b1);
    $and1 = $b1['filter']['and'];
    $this->assertContains(['objectType' => 'Glossary terms'], $and1);
    $this->assertContains(['entityType' => 'AtlasGlossaryTerm'], $and1);
    $this->assertContains(['glossary' => 'Finance'], $and1);

    // Assert second request body offset advanced.
    $b2 = json_decode((string) $history[1]['request']->getBody(), TRUE);
    $this->assertSame(3, $b2['offset']);
  }

  /**
   * Glossary 'all' should NOT add a glossary filter term.
   */
  public function testSearchGlossaryTermsAllSkipsGlossaryFilter(): void {
    $history = [];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode(['value' => []])),
    ], $history);

    $sut = $this->buildSut($http);
    $sut->searchGlossaryTerms('abc', 50, 'all');

    $body = json_decode((string) $history[0]['request']->getBody(), TRUE);
    $filters = $body['filter']['and'];
    foreach ($filters as $f) {
      $this->assertArrayNotHasKey('glossary', $f, 'No glossary filter when glossary=all');
    }
  }

  /**
   * Happy path: returns parsed JSON and hits expected URL.
   */
  public function testGetTermMetadataHappyPath(): void {
    $history = [];
    $payload = ['entity' => ['guid' => 'abcd-1234']];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($payload)),
    ], $history);

    $sut = $this->buildSut($http);
    $out = $sut->getTermMetadata('abcd-1234');

    $this->assertIsArray($out);
    $this->assertSame('abcd-1234', $out['entity']['guid']);

    $req = $history[0]['request'];
    $this->assertSame('GET', $req->getMethod());
    $this->assertStringEndsWith('/catalog/api/atlas/v2/entity/guid/abcd-1234', (string) $req->getUri());
  }

  /**
   * Parses enum/array<enum>, sets types/options, skips disabled, caches/reuses.
   */
  public function testGetManagedAttributeDefinitionsParseAndCache(): void {
    $history = [];
    $typedefs = [
      'enumDefs' => [
        [
          'name' => 'StatusEnum',
          'elementDefs' => [
            ['value' => 'Active'],
            ['value' => 'Inactive'],
          ],
        ],
      ],
      'businessMetadataDefs' => [
        [
          'name' => 'MyGroup',
          'description' => 'Group desc',
          'attributeDefs' => [
            [
              'name' => 'Status',
              'typeName' => 'StatusEnum',
              'description' => 'Status field',
            ],
            [
              'name' => 'Tags',
              'typeName' => 'array<StatusEnum>',
              'description' => 'Tags field',
            ],
            [
              'name' => 'Free',
              'typeName' => 'string',
              'description' => 'Free text',
            ],
            [
              'name' => 'Disabled',
              'typeName' => 'StatusEnum',
              'options' => ['isDisabled' => 'true'],
            ],
          ],
        ],
      ],
    ];

    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($typedefs)),
    ], $history);

    $cache = $this->cacheBin();
    $sut = $this->buildSut($http, $cache);

    // First call populates cache.
    $out1 = $sut->getManagedAttributeDefinitions(FALSE);
    $this->assertIsArray($out1);
    $this->assertArrayHasKey('MyGroup.Status', $out1);
    $this->assertArrayHasKey('MyGroup.Tags', $out1);
    $this->assertArrayHasKey('MyGroup.Free', $out1);
    $this->assertArrayNotHasKey('MyGroup.Disabled', $out1, 'Disabled field skipped');

    $this->assertSame('enum', $out1['MyGroup.Status']['type']);
    $this->assertSame(['Active', 'Inactive'], $out1['MyGroup.Status']['options']);
    $this->assertSame('enum[]', $out1['MyGroup.Tags']['type']);
    $this->assertSame('freeform', $out1['MyGroup.Free']['type']);

    // Second call reuses cache (no second HTTP).
    $out2 = $sut->getManagedAttributeDefinitions(FALSE);
    $this->assertEquals($out1, $out2);
    $this->assertCount(1, $history);
  }

  /**
   * Bypass cache: TRUE forces a fresh HTTP request each time.
   */
  public function testGetManagedAttributeDefinitionsBypassCache(): void {
    $history = [];
    $typedefs1 = [
      'enumDefs' => [],
      'businessMetadataDefs' => [
        [
          'name' => 'G',
          'attributeDefs' => [
            ['name' => 'A', 'typeName' => 'string'],
          ],
        ],
      ],
    ];
    $typedefs2 = [
      'enumDefs' => [],
      'businessMetadataDefs' => [
        [
          'name' => 'G',
          'attributeDefs' => [
            ['name' => 'B', 'typeName' => 'string'],
          ],
        ],
      ],
    ];

    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($typedefs1)),
      new Response(200, ['Content-Type' => 'application/json'], json_encode($typedefs2)),
    ], $history);

    $sut = $this->buildSut($http);

    $a = $sut->getManagedAttributeDefinitions(TRUE);
    $b = $sut->getManagedAttributeDefinitions(TRUE);

    $this->assertArrayHasKey('G.A', $a);
    $this->assertArrayHasKey('G.B', $b);
    $this->assertCount(2, $history, 'Two HTTP calls when bypassing cache');
  }

  /**
   * No token short-circuits managed attribute typedefs to NULL (no HTTP).
   */
  public function testGetManagedAttributeDefinitionsNoToken(): void {
    $history = [];
    $http = $this->httpWithHistory([], $history);

    $sut = $this->buildSut($http, $this->cacheBin(), NULL);
    $this->assertNull($sut->getManagedAttributeDefinitions(FALSE));
    $this->assertCount(0, $history);
  }

  /**
   * Groups by definition group and copies labels/descriptions/decoded values.
   */
  public function testBuildGroupedManagedAttributes(): void {
    // Definitions keyed by "Group.Field".
    $defs = [
      'MyGroup.Status' => [
        'group' => 'MyGroup',
        'group_description' => 'Group Desc',
        'field' => 'Status',
        'description' => 'Status desc',
      ],
      'MyGroup.Tags' => [
        'group' => 'MyGroup',
        'group_description' => 'Group Desc',
        'field' => 'Tags',
        'description' => 'Tags desc',
      ],
    ];

    // Managed attributes as returned by Purview term:
    $attrs = [
      ['name' => 'MyGroup.Status', 'value' => json_encode(['Active'])],
      ['name' => 'MyGroup.Tags', 'value' => json_encode(['A', 'B'])],
      // Ignored (no def).
      ['name' => 'Unknown.Field', 'value' => json_encode(['x'])],
      // Ignored (no values).
      ['name' => 'MyGroup.Empty', 'value' => json_encode([])],
    ];

    $history = [];
    $http = $this->httpWithHistory([], $history);
    $sut = $this->buildSut($http);
    $grouped = $sut->buildGroupedManagedAttributes($attrs, $defs);

    $this->assertArrayHasKey('MyGroup', $grouped);
    $this->assertSame('Group Desc', $grouped['MyGroup']['description']);

    $fields = $grouped['MyGroup']['fields'];
    $this->assertSame(['Status', 'Tags'], array_column($fields, 'label'));
    $this->assertSame(['Status desc', 'Tags desc'], array_column($fields, 'description'));
    $this->assertSame([['Active'], ['A', 'B']], array_column($fields, 'values'));
  }

  // ---------- Helpers ----------

  /**
   * Builds the Subject Under Test (SUT) for PurviewClassicTypesApiClient.
   *
   * Creates an instance wired with:
   *  - a provided HTTP client,
   *  - a ConfigFactory that serves drupal_purview.settings:rest_endpoint,
   *  - a mocked auth service that returns $token (or NULL for no token),
   *  - a logger factory that returns a mocked logger channel,
   *  - a cache backend (defaults to in-memory if none is supplied).
   *
   * @param \GuzzleHttp\ClientInterface $http
   *   HTTP client used by the API client.
   * @param \Drupal\Core\Cache\CacheBackendInterface|null $cache
   *   Cache backend used for responses. If NULL, an in-memory backend is used.
   * @param string|null $token
   *   Access token returned by the mocked auth service. Pass NULL to simulate
   *   the "no token" short-circuit behavior.
   * @param string $endpoint
   *   Base REST endpoint (e.g., "https://example.purview"). Used by the mocked
   *   ConfigFactory for drupal_purview.settings:rest_endpoint.
   *
   * @return \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
   *   The fully constructed SUT.
   */
  private function buildSut(
    ClientInterface $http,
    ?CacheBackendInterface $cache = NULL,
    ?string $token = 'token-123',
    string $endpoint = 'https://example.purview',
  ): PurviewClassicTypesApiClient {
    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getAccessToken'])
      ->getMock();
    $auth->method('getAccessToken')->willReturn($token);

    $logger = $this->createMock(LoggerInterface::class);

    return new PurviewClassicTypesApiClient(
      $this->configFactory($endpoint),
      $http,
      $this->loggerFactory($logger),
      $auth,
      $cache ?? $this->cacheBin(),
    );
  }

  /**
   * Build a Guzzle client with response queue and capture request history.
   *
   * @param \Psr\Http\Message\ResponseInterface[] $responses
   *   Responses to return, in order.
   * @param array<int,array> $history
   *   Will be populated with request/response history.
   */
  private function httpWithHistory(array $responses, array &$history): ClientInterface {
    $mock = new MockHandler($responses);
    $stack = HandlerStack::create($mock);
    $stack->push(Middleware::history($history));
    return new Client(['handler' => $stack, 'http_errors' => TRUE]);
  }

  /**
   * Minimal config factory that returns only rest_endpoint.
   *
   * @param string $endpoint
   *   Base REST endpoint to return for the 'rest_endpoint' config key.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   The mocked config factory.
   */
  private function configFactory(string $endpoint = 'https://example.purview'): ConfigFactoryInterface {
    $config = $this->createMock(Config::class);
    $config->method('get')->willReturnMap([
      ['rest_endpoint', $endpoint],
    ]);
    $factory = $this->createMock(ConfigFactoryInterface::class);
    $factory->method('get')->with('drupal_purview.settings')->willReturn($config);
    return $factory;
  }

  /**
   * Logger factory that returns a provided logger.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger to return for the 'drupal_purview' channel.
   *
   * @return \Drupal\Core\Logger\LoggerChannelFactoryInterface
   *   The mocked logger factory.
   */
  private function loggerFactory(LoggerInterface $logger): LoggerChannelFactoryInterface {
    $factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $factory->method('get')->with('drupal_purview')->willReturn($logger);
    return $factory;
  }

  /**
   * Provides a cache backend for unit tests.
   *
   * - If $useRealCache is FALSE, returns a NullBackend so cache operations are
   *   effectively no-ops. Useful in tests where caching is not required.
   * - If $useRealCache is TRUE, returns a MemoryBackend seeded with a mocked
   *   TimeInterface. This avoids instantiating the real Time service (which now
   *   requires constructor arguments) while still allowing cache TTL/expiry
   *   logic to be exercised in tests.
   *
   * @param bool $useRealCache
   *   TRUE to return a MemoryBackend with mocked Time; FALSE for a NullBackend.
   * @param int $t
   *   Fixed timestamp used by the mocked TimeInterface (default 1_700_000_000).
   *
   * @return \Drupal\Core\Cache\CacheBackendInterface
   *   A cache backend suitable for unit tests.
   */
  private function cacheBin(bool $useRealCache = TRUE, int $t = 1_700_000_000): CacheBackendInterface {
    if (!$useRealCache) {
      // For tests that don't assert cache behaviour.
      return new NullBackend('drupal_purview');
    }

    // Mock TimeInterface instead of constructing Time().
    $time = $this->createMock(TimeInterface::class);
    $time->method('getRequestTime')->willReturn($t);
    $time->method('getCurrentTime')->willReturn($t);
    $time->method('getRequestMicroTime')->willReturn((float) $t);

    return new MemoryBackend($time, 'drupal_purview');
  }

}
