<?php

declare(strict_types=1);

namespace Drupal\Tests\drupal_purview\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\MemoryBackend;
use Drupal\Core\Cache\NullBackend;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\drupal_purview\Service\PurviewAuthenticationService;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;

/**
 * Extra behavior tests for PurviewGovernanceDomainApiClient.
 *
 * - request shaping (owners/attributes, IDs payload)
 * - transformations (icon mapping, sorting)
 * - filtering + dedupe + caching
 * - simple aggregation math.
 *
 * @group drupal_purview
 * @covers \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
 */
final class PurviewGovernanceDomainApiClientBehaviorTest extends UnitTestCase {

  // ---------- Tests ----------

  /**
   * Maps asset type to icon and sorts by name.
   *
   * GIVEN a response with two assets (powerbi_report, unknown).
   * WHEN getDataAssetMetadataByIds() is invoked.
   * THEN results are sorted by name ascending and icons are mapped.
   */
  public function testDataAssetMetadataByIdsMapsIconsAndSorts(): void {
    $history = [];
    $value = [
      ['name' => 'b-report', 'source' => ['assetType' => 'powerbi_report']],
      ['name' => 'a-unknown', 'source' => ['assetType' => 'something_else']],
    ];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode(['value' => $value])),
    ], $history);

    $sut = $this->buildSut($http);
    $out = $sut->getDataAssetMetadataByIds([['entityId' => '1'], ['entityId' => '2']]);

    $this->assertSame(['a-unknown', 'b-report'], array_column($out, 'name'), 'Sorted by name asc');
    $this->assertSame(['default', 'powerbi'], array_column($out, 'icon'), 'Icons mapped');
    // Also assert the POST contained the IDs.
    $body = json_decode((string) $history[0]['request']->getBody(), TRUE);
    $this->assertSame(['1', '2'], $body['ids']);
  }

  /**
   * Filters “Related”, dedupes, and caches results.
   *
   * Only 'Related' items with unique entityIds are returned. A second call
   * should hit cache and avoid a second HTTP request.
   */
  public function testRelatedEntityIdsForTermFiltersDedupeAndCaches(): void {
    $history = [];
    $payload = [
      'value' => [
        ['relationshipType' => 'Related', 'entityId' => 'X'],
        ['relationshipType' => 'Other', 'entityId' => 'Y'],
        // Duplicate.
        ['relationshipType' => 'Related', 'entityId' => 'X'],
      ],
    ];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($payload)),
      new Response(500, [], 'should-not-be-called'),
    ], $history);

    $sut = $this->buildSut($http);

    $first = $sut->getRelatedEntityIdsForTerm('term-1', 'DataAsset', 'Related', FALSE);
    $second = $sut->getRelatedEntityIdsForTerm('term-1', 'DataAsset', 'Related', FALSE);

    $this->assertSame(['X'], $first);
    $this->assertSame($first, $second, 'Cached result reused');
    $this->assertCount(1, $history, 'Only one HTTP request due to caching');
  }

  /**
   * Shapes owner/attribute filters for getDataProductsByDomain().
   *
   * GIVEN owner='owner-1' and attribute='status|Active'.
   * WHEN getDataProductsByDomain() is invoked (bypass cache).
   * THEN POST body includes owners and a managedAttributes entry with the
   * expected field/operator/type/value.
   */
  public function testGetDataProductsByDomainShapesOwnersAndAttributes(): void {
    $history = [];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode(['value' => []])),
    ], $history);

    $sut = $this->buildSut($http);
    $sut->getDataProductsByDomain('D123', 'kw', 'owner-1', 'status|Active', TRUE);

    $req = $history[0]['request'];
    $this->assertSame('POST', $req->getMethod());
    $body = json_decode((string) $req->getBody(), TRUE);

    $this->assertSame(['D123'], $body['domainIds']);
    $this->assertSame('Published', $body['status']);
    $this->assertSame('kw', $body['nameKeyword']);
    $this->assertSame(['owner-1'], $body['owners']);
    $this->assertNotEmpty($body['managedAttributes']);

    $attr = $body['managedAttributes'][0];
    $this->assertSame('status', $attr['field']);
    $this->assertSame('eq-any', $attr['operator']);
    $this->assertSame('multiChoice', $attr['type']);
    $this->assertSame(['Active'], $attr['value']);
  }

  /**
   * Computes and rounds average score to one decimal place.
   *
   * Optional domain filter is applied when provided.
   */
  public function testAverageDataQualityScoreByKeyFiltersAndRounds(): void {
    $history = [];
    $report = [
      ['businessDomainId' => 'D1', 'dataProductId' => 'A', 'score' => 0.9],
      ['businessDomainId' => 'D1', 'dataProductId' => 'A', 'score' => 0.8],
      ['businessDomainId' => 'D2', 'dataProductId' => 'B', 'score' => 1.0],
    ];
    $http = $this->httpWithHistory([
      new Response(200, ['Content-Type' => 'application/json'], json_encode($report)),
    ], $history);

    $sut = $this->buildSut($http);
    $avg = $sut->getAverageDataQualityScoreByKey('dataProductId', 'A', 'D1');

    $this->assertSame(85.0, $avg);
  }

  // ---------- Helpers ----------

  /**
   * Build the SUT with pluggable HTTP/cache and a token (or null).
   */
  private function buildSut(
    ClientInterface $http,
    ?MemoryBackend $cache = NULL,
    ?string $token = 'token-123',
  ): PurviewGovernanceDomainApiClient {
    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getAccessToken'])
      ->getMock();
    $auth->method('getAccessToken')->willReturn($token);

    return new PurviewGovernanceDomainApiClient(
      $this->configFactory(),
      $http,
      $this->loggerFactory(),
      $auth,
      $cache ?? $this->cacheBin(),
      $this->translator(),
    );
  }

  /**
   * Build a Guzzle client with a response queue and capture request history.
   *
   * @param \Psr\Http\Message\ResponseInterface[] $responses
   *   Responses to return, in order.
   * @param array<int,array> $history
   *   Will be filled with request/response history.
   */
  private function httpWithHistory(array $responses, array &$history): ClientInterface {
    $mock = new MockHandler($responses);
    $stack = HandlerStack::create($mock);
    $stack->push(Middleware::history($history));
    return new Client(['handler' => $stack, 'http_errors' => TRUE]);
  }

  /**
   * Minimal config factory returning Purview account + API version.
   */
  private function configFactory(string $account = 'acct123', string $apiVersion = '2023-10-01-preview'): ConfigFactoryInterface {
    $config = $this->createMock(Config::class);
    $config->method('get')->willReturnMap([
      ['purview_account_guid', $account],
      ['api_version', $apiVersion],
    ]);
    $factory = $this->createMock(ConfigFactoryInterface::class);
    $factory->method('get')->with('drupal_purview.settings')->willReturn($config);
    return $factory;
  }

  /**
   * Logger factory that returns the provided channel.
   */
  private function loggerFactory(?LoggerChannelInterface $channel = NULL): LoggerChannelFactoryInterface {
    $channel ??= $this->createMock(LoggerChannelInterface::class);
    $factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $factory->method('get')->with('drupal_purview')->willReturn($channel);
    return $factory;
  }

  /**
   * Light translator stub: returns plain strings for ->t() and plural.
   */
  private function translator(): TranslationInterface {
    $t = $this->createMock(TranslationInterface::class);

    // Used by TranslatableMarkup::render().
    $t->method('translateString')
      ->willReturnCallback(function ($string, array $args = [], array $options = []) {
        return strtr((string) $string, array_map('strval', $args));
      });

    // Sometimes called directly by code using the interface.
    $t->method('translate')
      ->willReturnCallback(function ($string, array $args = [], array $options = []) {
        return strtr((string) $string, array_map('strval', $args));
      });

    // Provide a minimal pluralization; include @count in replacements.
    $t->method('formatPlural')
      ->willReturnCallback(function ($count, $singular, $plural, array $args = [], array $options = []) {
        $chosen = ($count == 1) ? $singular : $plural;
        $args = ['@count' => $count] + $args;
        return strtr((string) $chosen, array_map('strval', $args));
      });

    return $t;
  }

  /**
   * Provides a cache backend for unit tests.
   *
   * - If $useRealCache is FALSE, returns a NullBackend so cache operations are
   *   effectively no-ops. Useful in tests where caching not needed.
   * - If $useRealCache is TRUE, returns a MemoryBackend seeded with a
   *   mocked TimeInterface. This avoids instantiating the real Time service
   *   (which now requires constructor arguments) while still allowing cache
   *   TTL/expiry logic to be exercised in tests.
   *
   * @param bool $useRealCache
   *   TRUE to return a MemoryBackend with mocked Time; FALSE for a NullBackend.
   * @param int $t
   *   Fixed timestamp used by the mocked TimeInterface (default 1_700_000_000).
   *
   * @return \Drupal\Core\Cache\MemoryBackend
   *   A cache backend suitable for unit tests.
   */
  private function cacheBin(bool $useRealCache = TRUE, int $t = 1_700_000_000): MemoryBackend {
    if (!$useRealCache) {
      // For tests that don't assert cache behaviour.
      return new NullBackend('drupal_purview');
    }

    // Mock TimeInterface instead of constructing Time().
    $time = $this->createMock(TimeInterface::class);
    $time->method('getRequestTime')->willReturn($t);
    $time->method('getCurrentTime')->willReturn($t);
    $time->method('getRequestMicroTime')->willReturn((float) $t);

    return new MemoryBackend($time, 'drupal_purview');
  }

}
