<?php

namespace Drupal\purview_unified_catalog_ui\Breadcrumb;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Url;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;

/**
 * Builds custom breadcrumbs for the data product metadata page.
 *
 * Replaces the default breadcrumb with a custom trail:
 *   Home > Data catalog > [Product Title]
 * The final breadcrumb is rendered as plain text (not a link).
 */
class DataProductBreadcrumbBuilder implements BreadcrumbBuilderInterface {

  /**
   * The Purview Governance Domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $apiClient;

  /**
   * Constructs a new DataProductBreadcrumbBuilder.
   *
   * @param \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient $apiClient
   *   The Purview API client for retrieving product metadata.
   */
  public function __construct(PurviewGovernanceDomainApiClient $apiClient) {
    $this->apiClient = $apiClient;
  }

  /**
   * Determines whether this breadcrumb builder applies to the current route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   *
   * @return bool
   *   TRUE if this builder should handle the route, FALSE otherwise.
   */
  public function applies(RouteMatchInterface $route_match): bool {
    return $route_match->getRouteName() === 'purview_unified_catalog_ui.product_metadata_page';
  }

  /**
   * Builds the breadcrumb for the data product metadata page.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   *
   * @return \Drupal\Core\Breadcrumb\Breadcrumb
   *   The constructed breadcrumb object.
   */
  public function build(RouteMatchInterface $route_match): Breadcrumb {
    $breadcrumb = new Breadcrumb();

    $breadcrumb->addLink(Link::createFromRoute(t('Home'), '<front>'));
    $breadcrumb->addLink(Link::createFromRoute(t('Data catalog'), 'purview_unified_catalog_ui.products_search'));

    // Get GUID and lookup product.
    $guid = $route_match->getRawParameter('guid');
    if ($guid) {
      $product = $this->apiClient->getDataProductMetadata($guid);
      if (!empty($product['name'])) {
        $breadcrumb->addLink(Link::fromTextAndUrl($product['name'], Url::fromRoute('<none>')));
        $breadcrumb->addCacheTags(["purview_product:$guid"]);
      }
    }

    // Cache context for the URL to prevent bleeding across routes.
    $breadcrumb->addCacheContexts(['url']);

    return $breadcrumb;
  }

}
