<?php

namespace Drupal\drupal_purview\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides methods to interact with Microsoft Purview Governance Domain APIs.
 */
class PurviewGovernanceDomainApiClient {

  use StringTranslationTrait;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The authentication service.
   *
   * @var \Drupal\drupal_purview\Service\PurviewAuthenticationService
   */
  protected PurviewAuthenticationService $authService;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The translation interface.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected TranslationInterface $translationInterface;

  /**
   * Constructs a new PurviewGovernanceDomainApiClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\drupal_purview\Service\PurviewAuthenticationService $authService
   *   The authentication service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translationInterface
   *   The translation interface.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    PurviewAuthenticationService $authService,
    CacheBackendInterface $cache,
    TranslationInterface $translationInterface,
  ) {
    $this->configFactory = $configFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('drupal_purview');
    $this->authService = $authService;
    $this->cache = $cache;
    $this->stringTranslation = $translationInterface;
  }

  /**
   * Retrieves facet values from the Purview catalog for a given resource type.
   *
   * @param string $resource
   *   The resource to query facets for (e.g., 'dataproducts', 'glossaryterms').
   * @param array $facet_names
   *   An array of facet names to retrieve (e.g., ['owner'], ['status']).
   * @param bool $bypass_cache
   *   Whether to bypass the cache and force a fresh request.
   *
   * @return array|null
   *   An associative array of facets and their values, or NULL on failure.
   */
  public function getFacetValues(string $resource, array $facet_names, bool $bypass_cache = FALSE): ?array {
    if (empty($resource) || empty($facet_names)) {
      return NULL;
    }

    // Build a unique cache ID based on resource and facet names.
    $cid = 'purview_facets:' . $resource . ':' . md5(implode(',', $facet_names));

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');

    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/{$resource}/facets";

    $facets = array_map(fn($name) => ['name' => $name], $facet_names);

    try {
      $data = [
        'nameKeyword' => '',
        'facets' => $facets,
      ];

      $response = $this->httpClient->request('POST', $url, [
        'body' => json_encode($data),
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Content-Type' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      $facets_result = $data['facets'] ?? NULL;

      if ($facets_result !== NULL) {
        $this->cache->set($cid, $facets_result, strtotime('+1 day'), ['purview_facets']);
      }

      return $facets_result;

    }
    catch (\Exception $e) {
      $this->logger->error('Purview facet request failed for @resource: @message', [
        '@resource' => $resource,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves the list of governance domains.
   *
   * @param bool $bypass_cache
   *   Whether to bypass the cache and force a fresh request.
   *
   * @return array|null
   *   An array of governance domains or NULL on failure.
   */
  public function getGovernanceDomains(bool $bypass_cache = FALSE): ?array {
    if (!$bypass_cache) {
      $cache = $this->cache->get('purview_governance_domains');
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/businessdomains";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $body = (string) $response->getBody();
      $data = json_decode($body, TRUE);

      // Guard against malformed or non-array JSON.
      if (JSON_ERROR_NONE !== json_last_error() || !is_array($data)) {
        $this->logger->error('Failed to decode Purview response JSON for governance domains.', [
          'json_error' => json_last_error_msg(),
          'body_sample' => mb_substr($body, 0, 200),
          'endpoint' => $url,
        ]);
        return NULL;
      }

      // Guard against expected "value" list,  missing or invalid.
      if (!isset($data['value']) || !is_array($data['value'])) {
        $this->logger->error('Purview response missing expected "value" array for governance domains.', [
          'keys' => implode(',', array_keys($data)),
          'endpoint' => $url,
        ]);
        return NULL;
      }

      $response = $data['value'];

      // Cache result for 1 day with a stable tag for easy invalidation.
      $this->cache->set(
        'purview_governance_domains',
        $response,
        strtotime('+1 day'),
        ['purview_governance_domains']
      );

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve governance domains: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Retrieves metadata for a specific Governance Domain.
   *
   * @param string $domain_id
   *   The unique ID of the Governance Domain.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of metadata for the domain, or NULL on failure.
   */
  public function getGovernanceDomainMetadata(string $domain_id, bool $bypass_cache = FALSE): ?array {
    if (empty($domain_id)) {
      return NULL;
    }

    $cid = 'purview_governance_domain:' . $domain_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/businessdomains/{$domain_id}";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if ($data !== NULL) {
        $this->cache->set($cid, $data, strtotime('+1 day'), ['purview_governance_domains']);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve metadata for Governance Domain with ID @id: @error', [
        '@id' => $domain_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves health actions for a specific Governance Domain.
   *
   * @param string $domain_id
   *   The unique ID of the governance domain.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of health actions for the domain, or NULL on failure.
   */
  public function getGovernanceDomainHealthActionsCount(string $domain_id, bool $bypass_cache = FALSE): ?int {
    if (empty($domain_id)) {
      return NULL;
    }

    $cid = 'purview_health_actions_count:' . $domain_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $api_version = $config->get('api_version');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/health/actions/query?api-version={$api_version}";

    try {
      $data = [
        "filters" => [
          "domainIds" => [$domain_id],
          "status" => [
            "NotStarted",
            "InProgress",
          ],
        ],
        "pageSize" => 0,
      ];

      $response = $this->httpClient->post($url, [
        'body' => json_encode($data),
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $count = json_decode($response->getBody()->getContents(), TRUE)['count'] ?? NULL;

      if ($count !== NULL) {
        $this->cache->set($cid, $count, strtotime('+1 hour'), ['purview_health_actions']);
      }

      return $count;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve health actions for Governance Domain ID @id: @error', [
        '@id' => $domain_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Builds a hierarchy of governance domains.
   *
   * @param array $domains
   *   The raw output from getGovernanceDomains().
   *
   * @return array
   *   The nested governance domain tree.
   */
  public function buildGovernanceDomainHierarchy(array $domains): array {
    $items = [];

    // Index domains by their unique refName.
    foreach ($domains as $domain) {
      $id = $domain['id'];
      $domain['children'] = [];
      $items[$id] = $domain;
    }

    // Link children to parents.
    $tree = [];
    foreach ($items as $ref_name => &$domain) {
      $parent_ref = $domain['parentId'] ?? NULL;

      if ($parent_ref && isset($items[$parent_ref])) {
        $items[$parent_ref]['children'][] = &$domain;
      }
      else {
        $tree[] = &$domain;
      }
    }
    unset($domain);

    // Sort recursively.
    $this->sortHierarchyByName($tree);

    return $tree;
  }

  /**
   * Format nested governance domain hierarchy into a flat list.
   *
   * Recursively traverses the domain hierarchy and appends each domain's ID and
   * name to the given options array, preserving hierarchy order. Indents child
   * domains using a prefix to visually represent nesting in the UI.
   *
   * @param array $domains
   *   The nested array of governance domains, each with 'id', 'name',
   *   and optionally 'children'.
   * @param array &$options
   *   The associative array of options to populate a select field.
   * @param int $depth
   *   The current depth in the hierarchy (used for indentation). Defaults to 0.
   */
  public function formatHierarchySelectList(array $domains, array &$options, int $depth = 0): void {
    foreach ($domains as $domain) {
      $prefix = str_repeat('--', $depth);
      $name = $domain['name'] ?? '';
      $id = $domain['id'];

      if ($name && $id) {
        $options[$id] = $prefix . $name;
      }

      if (!empty($domain['children'])) {
        $this->formatHierarchySelectList($domain['children'], $options, $depth + 1);
      }
    }
  }

  /**
   * Recursively sort domains by name.
   */
  protected function sortHierarchyByName(array &$tree): void {
    usort($tree, function ($a, $b) {
      return strcasecmp($a['name'], $b['name']);
    });

    foreach ($tree as &$item) {
      if (!empty($item['children'])) {
        $this->sortHierarchyByName($item['children']);
      }
    }
    unset($item);
  }

  /**
   * Retrieves the parent domain name for a given domain ID.
   *
   * @param string $domain_id
   *   The unique ID of the domain.
   *
   * @return string|null
   *   The name of the parent domain, or NULL if not found.
   */
  public function getDomainParent(string $domain_id): ?string {
    $domains = $this->getGovernanceDomains();
    if (!$domains) {
      return NULL;
    }

    // Build the hierarchy tree.
    $hierarchy = $this->buildGovernanceDomainHierarchy($domains);

    // Flatten the tree to find the target domain and its parent.
    $flat = [];
    $this->flattenHierarchy($hierarchy, NULL, $flat);

    // Find the parent name.
    return $flat[$domain_id]['parent_name'] ?? NULL;
  }

  /**
   * Flatten hierarchy into an array of [id => [name, parent_name]].
   *
   * @param array $nodes
   *   The hierarchy nodes.
   * @param string|null $parent_name
   *   The parent name if any.
   * @param array &$flat
   *   The flat output array.
   */
  protected function flattenHierarchy(array $nodes, ?string $parent_name, array &$flat): void {
    foreach ($nodes as $node) {
      $id = $node['id'] ?? $node['domains'][0]['id'] ?? NULL;
      $name = $node['name'] ?? NULL;

      if ($id && $name) {
        $flat[$id] = [
          'name' => $name,
          'parent_name' => $parent_name,
        ];
      }

      if (!empty($node['children'])) {
        $this->flattenHierarchy($node['children'], $name, $flat);
      }
    }
  }

  /**
   * Retrieves a list of Data Products for a specific domain.
   *
   * @param string $domain_id
   *   The domain's unique ID.
   * @param string $keywords
   *   The search keywords. Default is a blank search.
   * @param string $owner
   *   Search for data products by owner.
   * @param string $attribute
   *   The attribute to filter against.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh results.
   *
   * @return array|null
   *   An array of Data Products.
   */
  public function getDataProductsByDomain(string $domain_id, string $keywords = '', $owner = '', $attribute = '', bool $bypass_cache = FALSE): ?array {
    if (empty($domain_id)) {
      return NULL;
    }

    $cid_parts = [
      'purview_data_products',
      $domain_id,
      md5($keywords),
      $owner,
      $attribute,
    ];
    $cid = implode(':', $cid_parts);

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/dataproducts/query";

    try {
      $data = [
        'domainIds' => [
          $domain_id,
        ],
        'status' => 'Published',
        'nameKeyword' => $keywords,
        'skip' => 0,
        'top' => 100,
      ];

      if ($owner !== '' && $owner !== 'all') {
        $data['owners'] = [$owner];
      }

      if ($attribute !== '' && $attribute !== 'all') {
        [$field, $value] = array_pad(explode('|', $attribute, 2), 2, '');
        $field = trim($field);
        $value = trim($value);

        $data['managedAttributes'][] = [
          'field' => $field,
          'operator' => 'eq-any',
          'type' => 'multiChoice',
          'value' => [$value],
        ];
      }

      $response = $this->httpClient->post($url, [
        'body' => json_encode($data),
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      if ($result !== NULL) {
        $this->cache->set($cid, $result, strtotime('+1 day'), ['purview_data_products']);
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve Data Products for domain @domain: @error', [
        '@domain' => $domain_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Searches glossary terms across all domains.
   *
   * @param string $keywords
   *   The search keywords.
   * @param string $owner
   *   (Optional) Filter by owner GUID.
   * @param array $term_ids
   *   (optional) Filter by term IDs.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh results.
   *
   * @return array|null
   *   The result array of glossary terms, or NULL on failure.
   */
  public function searchGlossaryTerms(string $keywords = '', string $owner = '', array $term_ids = [], bool $bypass_cache = FALSE): ?array {
    $cid_parts = [
      'purview_glossary_terms_all',
      md5($keywords),
      $owner,
      !empty($term_ids) ? md5(implode(',', $term_ids)) : '',
    ];
    $cid = implode(':', $cid_parts);

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/terms/query";

    try {
      $data = [
        'status' => 'Published',
        'nameKeyword' => $keywords,
        'skip' => 0,
        'top' => 100,
      ];

      if ($owner !== '' && $owner !== 'all') {
        $data['owners'] = [$owner];
      }

      if (!empty($term_ids)) {
        $data['ids'] = $term_ids;
      }

      $response = $this->httpClient->post($url, [
        'body' => json_encode($data),
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      if ($result !== NULL) {
        $this->cache->set($cid, $result, strtotime('+1 day'), ['purview_glossary_search']);
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to search glossary terms across all domains: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves a list of Glossary Terms for a specific domain.
   *
   * @param string $domain_id
   *   The domain's unique ID.
   * @param string $keywords
   *   The search keywords. Default is a blank search.
   * @param string $owner
   *   The owner to filter against.
   * @param string $attribute
   *   The attribute to filter against.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh results.
   *
   * @return array|null
   *   An array of Glossary Terms.
   */
  public function getGlossaryTermsByDomain(string $domain_id, string $keywords = '', $owner = '', $attribute = '', bool $bypass_cache = FALSE): ?array {
    if (empty($domain_id)) {
      return NULL;
    }

    $cid_parts = [
      'purview_glossary_terms',
      $domain_id,
      md5($keywords),
      $owner,
      $attribute,
    ];
    $cid = implode(':', $cid_parts);

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/terms/query";

    try {
      $data = [
        'domainIds' => [
          $domain_id,
        ],
        'status' => 'Published',
        'nameKeyword' => $keywords,
        'skip' => 0,
        'top' => 100,
      ];

      if ($owner !== '' && $owner !== 'all') {
        $data['owners'] = [$owner];
      }

      if ($attribute !== '' && $attribute !== 'all') {
        [$field, $value] = array_pad(explode('|', $attribute, 2), 2, '');
        $field = trim($field);
        $value = trim($value);

        $data['managedAttributes'][] = [
          'field' => $field,
          'operator' => 'eq-any',
          'type' => 'multiChoice',
          'value' => [$value],
        ];
      }

      $response = $this->httpClient->post($url, [
        'body' => json_encode($data),
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      if ($result !== NULL) {
        $this->cache->set($cid, $result, strtotime('+1 day'), ['purview_glossary_terms']);
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve Glossary Terms for domain @domain: @error', [
        '@domain' => $domain_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves all glossary terms associated with a data product.
   *
   * @param string $data_product_id
   *   The GUID of the data product.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of related glossary terms, or NULL on failure.
   */
  public function getGlossaryTermsForDataProduct(string $data_product_id, bool $bypass_cache = FALSE): ?array {
    $cid = 'purview:data_product_terms:' . $data_product_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/dataproducts/{$data_product_id}/relationships?entityType=Term";

    try {
      $response = $this->httpClient->request('GET', $url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      // Store in cache.
      if ($result !== NULL) {
        $this->cache->set($cid, $result, strtotime('+1 day'), ['data_product_terms']);
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve glossary terms for data product @id: @message', [
        '@id' => $data_product_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves a list of metadata for a given Data Products.
   *
   * @param string $product_id
   *   The product's unique ID.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of metadata for a given Data Product.
   */
  public function getDataProductMetadata(string $product_id, bool $bypass_cache = FALSE): ?array {
    if (empty($product_id)) {
      return NULL;
    }

    $cid = 'purview_data_product_metadata:' . $product_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/dataproducts/{$product_id}";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if ($data !== NULL) {
        $this->cache->set($cid, $data, strtotime('+1 day'), ['purview_data_products']);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve metadata for Data Product with ID @id: @error', [
        '@id' => $product_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves data assets related to a given data product.
   *
   * @param string $product_id
   *   The GUID of the data product.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of related data assets or NULL on failure.
   */
  public function getDataAssetsForDataProduct(string $product_id, bool $bypass_cache = FALSE): ?array {
    if (empty($product_id)) {
      return NULL;
    }

    $cid = 'purview_data_assets_for_product:' . $product_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $url = "https://api.purview-service.microsoft.com/datagovernance/catalog/dataproducts/{$product_id}/relationships?entityType=DataAsset";

    try {
      $response = $this->httpClient->request('GET', $url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if ($data !== NULL) {
        $this->cache->set($cid, $data, strtotime('+1 day'), ['purview_data_products', 'purview_data_assets']);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get data assets for data product @guid: @message', [
        '@guid' => $product_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Queries multiple data assets by ID.
   *
   * @param array $ids
   *   An array of data asset IDs (GUIDs).
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of data assets or NULL on failure.
   */
  public function getDataAssetMetadataByIds(array $ids, bool $bypass_cache = FALSE): ?array {
    if (empty($ids)) {
      return NULL;
    }

    // Extract and sort entity IDs for a consistent cache key.
    $id_array = array_map(fn($item) => $item['entityId'], $ids);
    sort($id_array);
    $hash = md5(implode(',', $id_array));
    $cid = 'purview_data_asset_metadata_by_ids:' . $hash;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/dataassets/query";

    // Prepare IDs.
    $id_array = [];
    foreach ($ids as $id) {
      $id_array[] = $id['entityId'];
    }

    try {
      $response = $this->httpClient->request('POST', $url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
        'body' => json_encode(['ids' => $id_array]),
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if (isset($data['value']) && is_array($data['value'])) {
        foreach ($data['value'] as &$asset) {
          $asset['icon'] = $this->mapAssetTypeToIcon($asset['source']['assetType']);
        }

        $sorted = $this->sortDataAssetsByName($data['value']);
        $this->cache->set($cid, $sorted, strtotime('+1 hour'), ['purview_data_products', 'purview_data_assets']);
        return $sorted;
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to query data assets by IDs: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets related entity IDs for a glossary term.
   *
   * @param string $term_guid
   *   The glossary term GUID.
   * @param string $entity_type
   *   The entity type to retrieve.
   * @param string $relationship_type
   *   The relationship type to include (default: 'Related').
   * @param bool $bypass_cache
   *   If TRUE, skip cache and hit the API.
   *
   * @return string[]
   *   A list of related entity GUIDs (strings). Empty array if none/failure.
   */
  public function getRelatedEntityIdsForTerm(string $term_guid, string $entity_type, string $relationship_type = 'Related', bool $bypass_cache = FALSE): array {
    if ($term_guid === '') {
      return [];
    }

    $cid = "purview_term_relationships:{$term_guid}:{$entity_type}";
    if (!$bypass_cache) {
      if ($cached = $this->cache->get($cid)) {
        return is_array($cached->data) ? $cached->data : [];
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return [];
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/terms/{$term_guid}/relationships?entityType={$entity_type}";

    try {
      $res = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($res->getBody()->getContents(), TRUE);
      $rows = is_array($data['value'] ?? NULL) ? $data['value'] : [];

      // Extract related entityIds.
      $ids = [];
      foreach ($rows as $row) {
        if (($row['relationshipType'] ?? '') === $relationship_type && !empty($row['entityId'])) {
          $ids[] = (string) $row['entityId'];
        }
      }

      // Normalize: unique + reindex.
      $ids = array_values(array_unique($ids));

      // Cache for a day; tag by term + relationships.
      if (!empty($ids)) {
        $this->cache->set($cid, $ids, strtotime('+1 day'), ['purview_terms', 'purview_term_relationships']);
      }

      return $ids;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to fetch term relationships for @term: @msg', [
        '@term' => $term_guid,
        '@msg' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Sorts an array of data assets by their 'name' key.
   *
   * @param array $assets
   *   The unsorted data assets.
   *
   * @return array
   *   The sorted data assets.
   */
  protected function sortDataAssetsByName(array $assets): array {
    usort($assets, function ($a, $b) {
      return strcasecmp($a['name'] ?? '', $b['name'] ?? '');
    });
    return $assets;
  }

  /**
   * Retrieves a list of metadata for a given Glossary Term.
   *
   * @param string $term_id
   *   The term's unique ID.
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An array of metadata for a given Glossary Term.
   */
  public function getGlossaryTermMetadata(string $term_id, bool $bypass_cache = FALSE): ?array {
    if (empty($term_id)) {
      return NULL;
    }

    $cid = 'purview_glossary_term_metadata:' . $term_id;

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/catalog/terms/{$term_id}";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if ($data !== NULL) {
        $this->cache->set($cid, $data, strtotime('+1 day'), ['purview_glossary_terms', 'purview_glossary_term_metadata']);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve metadata for Glossary Term with ID @id: @error', [
        '@id' => $term_id,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets the Data Quality Report for a Business Domain.
   *
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   The data quality report, or NULL on failure.
   */
  public function getDataQualityReport(bool $bypass_cache = FALSE): ?array {
    $cid = 'purview_data_quality_report';

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $purview_account_guid = $config->get('purview_account_guid');
    $url = "https://{$purview_account_guid}-api.purview-service.microsoft.com/datagovernance/quality/business-domains/business-domain-id/report?api-version=2023-10-01-preview";

    try {
      $response = $this->httpClient->request('GET', $url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if ($data !== NULL) {
        $this->cache->set($cid, $data, strtotime('+1 hour'), ['purview_data_quality_report']);
      }

      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get data quality report: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Calculates the average data quality score filtered by key.
   *
   * @param string $key
   *   The key to filter by. Should be 'businessDomainId' or 'dataProductId'.
   * @param string $value
   *   The ID value to match against the key.
   * @param string|null $domain_id
   *   Optional: restrict to a specific domain ID as a second filter.
   *
   * @return float|null
   *   The average score as a percentage (0–100) with 1 decimal place,
   *   or NULL if no scores found.
   */
  public function getAverageDataQualityScoreByKey(string $key, string $value, ?string $domain_id = NULL): ?float {
    $report = $this->getDataQualityReport();

    if (!$report || !is_array($report)) {
      return NULL;
    }

    // Filter by primary key and optional domain_id.
    $filtered = array_filter($report, function ($row) use ($key, $value, $domain_id) {
      $matchesKey = isset($row[$key]) && $row[$key] === $value;
      $matchesDomain = $domain_id ? (isset($row['businessDomainId']) && $row['businessDomainId'] === $domain_id) : TRUE;
      return $matchesKey && $matchesDomain;
    });

    if (empty($filtered)) {
      return NULL;
    }

    // Sum and average the scores.
    $totalScore = array_reduce($filtered, function ($carry, $row) {
      return $carry + (float) $row['score'];
    }, 0);

    $average = $totalScore / count($filtered);
    return round($average * 100, 1);
  }

  /**
   * Returns a label for a given data quality percentage.
   *
   * @param float $score
   *   Percentage (0–100).
   *
   * @return string
   *   The quality label.
   */
  public function getDataQualityLabel(float $score): string {
    if ($score >= 90) {
      return $this->t('Healthy');
    }
    elseif ($score >= 75) {
      return $this->t('Fair');
    }
    else {
      return $this->t('Fail');
    }
  }

  /**
   * Retrieves the total count of Data Products for a given Governance Domain.
   *
   * @param string $domain_id
   *   The domain's unique ID.
   *
   * @return int|null
   *   The number of Data Products for the domain, or NULL on failure.
   */
  public function getDataProductCountForDomain(string $domain_id): ?int {
    $products = $this->getDataProductsByDomain($domain_id);
    return $products['count'] ?? 0;
  }

  /**
   * Retrieves the total count of Glossary Terms for a given Governance Domain.
   *
   * @param string $domain_id
   *   The domain's unique ID.
   *
   * @return int|null
   *   The number of Data Products for the domain, or NULL on failure.
   */
  public function getGlossaryTermCountForDomain(string $domain_id): ?int {
    $terms = $this->getGlossaryTermsByDomain($domain_id);
    return $terms['count'] ?? 0;
  }

  /**
   * Maps assetType to an icon filename.
   *
   * @param string $assetType
   *   The asset type.
   *
   * @return string
   *   The icon filename.
   */
  protected function mapAssetTypeToIcon(string $assetType): string {
    switch ($assetType) {
      case 'powerbi_table':
      case 'powerbi_dataset':
      case 'powerbi_report':
        return 'powerbi';

      case 'azure_synapse_dedicated_sql_table':
      case 'azure_synapse_dedicated_sql_view':
        return 'azure_synapse';

      default:
        return 'default';
    }
  }

}
