<?php

namespace Drupal\drupal_purview\Utility;

use Drupal\drupal_purview\Service\GraphApiClient;
use Drupal\Core\Template\TwigEnvironment;

/**
 * Provides helper utilities for the drupal_purview module.
 */
class PurviewHelper {

  /**
   * The Graph API client service.
   *
   * @var \Drupal\drupal_purview\Service\GraphApiClient
   */
  protected GraphApiClient $client;

  /**
   * The Twig environment for rendering templates.
   *
   * @var \Drupal\Core\Template\TwigEnvironment
   */
  protected TwigEnvironment $twig;

  /**
   * Constructs a new PurviewHelper instance.
   *
   * @param \Drupal\drupal_purview\Service\GraphApiClient $client
   *   The Microsoft Graph API client service.
   * @param \Drupal\Core\Template\TwigEnvironment $twig
   *   The Twig environment service.
   */
  public function __construct(GraphApiClient $client, TwigEnvironment $twig) {
    $this->client = $client;
    $this->twig = $twig;
  }

  /**
   * Generates two HSL color shades based on a letters.
   *
   * @param string $letters
   *   The first letter or first two letters (initials).
   *
   * @return array
   *   An array with 'border' and 'background' colors.
   */
  public static function generateLetterColors(string $letters): array {
    $letters = strtoupper($letters);

    $base = 0;
    $shift = 0;

    if (strlen($letters) >= 1 && ctype_alpha($letters[0])) {
      $base = ord($letters[0]) - ord('A');
    }

    if (strlen($letters) >= 2 && ctype_alpha($letters[1])) {
      $shift = ord($letters[1]) - ord('A');
    }

    // Base hue.
    $hue = ($base % 26) / 26 * 360;

    // Shift hue by second letter, shift up to 120 degrees.
    $hueShift = ($shift % 26) / 26 * 120;

    $hue = fmod($hue + $hueShift, 360);

    $border = "hsl($hue, 70%, 40%)";
    $background = "hsl($hue, 70%, 90%)";

    return [
      'border' => $border,
      'background' => $background,
    ];
  }

  /**
   * Converts a light HSL color to a darker variant for WCAG 2.1 AA.
   *
   * This is useful for ensuring sufficient contrast when overlaying white text
   * on background colors derived from HSL.
   *
   * @param string $hsl
   *   The input HSL color string, e.g. "hsl(120, 70%, 90%)".
   * @param float $darkness
   *   The desired lightness percentage for the darkened color (default: 30.0).
   *
   * @return string
   *   A modified HSL string with the specified lightness, or "#000" on failure.
   */
  public static function makeDarkColor(string $hsl, float $darkness = 30.0): string {
    // Example input: "hsl(120, 70%, 90%)".
    if (preg_match('/hsl\(([\d.]+),\s*([\d.]+)%,\s*([\d.]+)%\)/', $hsl, $matches)) {
      $hue = $matches[1];
      $saturation = $matches[2];
      // Force lightness to darkness %.
      return "hsl($hue, {$saturation}%, {$darkness}%)";
    }
    // Fallback: just return black.
    return "#000";
  }

  /**
   * Renders owner user objects using a Twig template.
   *
   * @param array $owners
   *   Array of arrays with 'id' keys.
   *
   * @return string
   *   Rendered HTML.
   */
  public function renderOwnersSmall(array $owners): string {
    $users = [];

    foreach ($owners as $owner) {
      $user = $this->client->getUserObjectFromGuid($owner['id']);
      if ($user) {
        // Generate initials.
        $name_parts = explode(' ', $user['displayName']);
        $initials = '';
        foreach ($name_parts as $part) {
          if (strlen($part) > 0) {
            $initials .= $part[0];
          }
        }
        $initials = strtoupper($initials);

        // Generate colors based on initials.
        $colors = self::generateLetterColors($initials);
        $color = self::makeDarkColor($colors['background'], 40);

        $users[] = [
          'displayName' => $user['displayName'],
          'mail' => $user['mail'],
          'initials' => $initials,
          'color' => $color,
        ];
      }
    }

    return $this->twig->render('@drupal_purview/purview-users--small.html.twig', [
      'users' => $users,
    ]);
  }

  /**
   * Renders owner user objects horizontally using a Twig template.
   *
   * @param array $owners
   *   Array of arrays with 'id' keys.
   *
   * @return string
   *   Rendered HTML.
   */
  public function renderOwnersHorizontal(array $owners): string {
    $users = [];

    foreach ($owners as $owner) {
      $user = $this->client->getUserObjectFromGuid($owner['id']);
      if ($user) {
        $name = $user['displayName'] ?? '';
        $email = $user['mail'] ?? '';
        $initials = strtoupper(implode('', array_map(fn($part) => $part[0] ?? '', explode(' ', $name))));
        $color = self::makeDarkColor(self::generateLetterColors($initials)['background'], 40);

        $users[] = [
          'name' => $name,
          'email' => $email,
          'initials' => $initials,
          'color' => $color,
        ];
      }
    }

    return $this->twig->render('@drupal_purview/purview-owners--horizontal.html.twig', [
      'users' => $users,
    ]);
  }

}
