<?php

declare(strict_types=1);

namespace Drupal\Tests\drupal_purview\Unit;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\drupal_purview\Service\PurviewAuthenticationService;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;

/**
 * Unit tests for PurviewGovernanceDomainApiClient.
 *
 * Core behavior covered:
 * - Happy path fetch + cache write on bypass
 * - Cache-hit short-circuit (no HTTP/auth; no cache set)
 * - 401/exception path => NULL + error log
 * - Malformed JSON => NULL + error log.
 *
 * These unit tests use Guzzle's MockHandler and lightweight doubles for
 * Config, Logger, Translation, and Cache; no kernel boot is required.
 *
 * @group drupal_purview
 * @covers \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
 */
final class PurviewGovernanceDomainApiClientTest extends UnitTestCase {

  // ---------- Tests ----------

  /**
   * Verifies happy path on cache bypass.
   *
   * GIVEN a valid token, a cache miss, and a 200 JSON response
   * WHEN getGovernanceDomains(TRUE) is invoked
   * THEN it returns the parsed array and writes the result to cache.
   */
  public function testGetGovernanceDomainsHappyPath(): void {
    $payload = json_encode([
      'value' => [
        ['id' => 'A', 'name' => 'Alpha'],
        ['id' => 'B', 'name' => 'Beta'],
      ],
    ]);

    $client = $this->makeClient([
      new Response(200, ['Content-Type' => 'application/json'], $payload),
    ]);

    $configFactory = $this->configFactoryWithGuid('abc123');

    $auth = $this->createMock(PurviewAuthenticationService::class);
    $auth->method('getAccessToken')->willReturn('fake.jwt');

    $cache = $this->createMock(CacheBackendInterface::class);
    // Simulate cache miss first call.
    $cache->method('get')->with('purview_governance_domains')->willReturn(FALSE);
    $cache->expects($this->once())->method('set')
      ->with('purview_governance_domains', $this->isType('array'), $this->greaterThan(time()), $this->isType('array'));

    $sut = $this->buildSut($client, $configFactory, $auth, $cache);

    $out = $sut->getGovernanceDomains(TRUE);
    $this->assertIsArray($out);
    $this->assertCount(2, $out);
    $this->assertSame('Alpha', $out[0]['name']);
  }

  /**
   * Verifies cache hit short-circuits HTTP and cache writes.
   *
   * GIVEN a populated cache entry for 'purview_governance_domains'
   * WHEN getGovernanceDomains(FALSE) is invoked
   * THEN it returns the cached value and does not call the auth service.
   */
  public function testGetGovernanceDomainsUsesCache(): void {
    $cached = [['id' => 'C', 'name' => 'Cached']];

    $client = $this->makeClient([
      // Would 500 if called; we want to prove it is NOT
      // used because of cache hit.
      new Response(500, ['Content-Type' => 'application/json'], '{"error":"boom"}'),
    ]);

    $configFactory = $this->configFactoryWithGuid('abc123');

    $auth = $this->createMock(PurviewAuthenticationService::class);
    $auth->expects($this->never())->method('getAccessToken');

    $cacheItem = (object) ['data' => $cached];

    $cache = $this->createMock(CacheBackendInterface::class);
    $cache->method('get')->with('purview_governance_domains')->willReturn($cacheItem);
    $cache->expects($this->never())->method('set');

    $sut = $this->buildSut($client, $configFactory, $auth, $cache);

    $out = $sut->getGovernanceDomains(FALSE);
    $this->assertSame($cached, $out);
  }

  /**
   * Verifies 401 errors return NULL and are logged.
   *
   * GIVEN a valid token and ClientException(401)
   * WHEN getGovernanceDomains(TRUE) is invoked
   * THEN it returns NULL and logs an error.
   */
  public function testGetGovernanceDomains401ReturnsNull(): void {
    // Simulate a 401 that throws.
    $request = new Request(
      'GET',
      'https://abc123-api.purview-service.microsoft.com/datagovernance/catalog/businessdomains'
    );
    $client = $this->makeClient([
      new ClientException('Unauthorized', $request, new Response(401, [], '{"error":"unauthorized"}')),
    ]);

    $configFactory = $this->configFactoryWithGuid('abc123');

    $auth = $this->createMock(PurviewAuthenticationService::class);
    $auth->method('getAccessToken')->willReturn('expired');

    $cache = $this->createMock(CacheBackendInterface::class);
    $cache->method('get')->with('purview_governance_domains')->willReturn(FALSE);

    $loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $loggerChannel->expects($this->atLeastOnce())->method('error');

    $sut = $this->buildSut($client, $configFactory, $auth, $cache, $loggerChannel);

    $this->assertNull($sut->getGovernanceDomains(TRUE));
  }

  /**
   * Verifies 401 errors return NULL and are logged.
   *
   * GIVEN a valid token and ClientException(401)
   * WHEN getGovernanceDomains(TRUE) is invoked
   * THEN it returns NULL and logs an error.
   */
  public function testGetGovernanceDomainsMalformedJsonReturnsNull(): void {
    $client = $this->makeClient([
      new Response(200, ['Content-Type' => 'application/json'], '{not valid json'),
    ]);

    $configFactory = $this->configFactoryWithGuid('abc123');

    $auth = $this->createMock(PurviewAuthenticationService::class);
    $auth->method('getAccessToken')->willReturn('fake.jwt');

    $cache = $this->createMock(CacheBackendInterface::class);
    $cache->method('get')->with('purview_governance_domains')->willReturn(FALSE);

    $loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $loggerChannel->expects($this->atLeastOnce())->method('error');

    $sut = $this->buildSut($client, $configFactory, $auth, $cache, $loggerChannel);

    $this->assertNull($sut->getGovernanceDomains(TRUE));
  }

  // ---------- Helpers ----------

  /**
   * Builds the Subject Under Test (SUT) with lightweight doubles.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   HTTP client used by the API client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory providing drupal_purview.settings.
   * @param \Drupal\drupal_purview\Service\PurviewAuthenticationService $auth
   *   Auth service returning a token.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   Cache backend used for responses.
   * @param \Drupal\Core\Logger\LoggerChannelInterface|null $loggerChannel
   *   Optional logger channel (defaults to a mock).
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface|null $loggerFactory
   *   Optional logger factory (defaults to a mock returning the channel).
   * @param \Drupal\Core\StringTranslation\TranslationInterface|null $translation
   *   Optional translation service used by StringTranslationTrait.
   *
   * @return \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   *   Fully constructed SUT instance.
   */
  private function buildSut(
    ClientInterface $client,
    ConfigFactoryInterface $configFactory,
    PurviewAuthenticationService $auth,
    CacheBackendInterface $cache,
    ?LoggerChannelInterface $loggerChannel = NULL,
    ?LoggerChannelFactoryInterface $loggerFactory = NULL,
    ?TranslationInterface $translation = NULL,
  ): PurviewGovernanceDomainApiClient {
    // Logger factory that returns a channel.
    $loggerChannel = $loggerChannel ?? $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $loggerFactory ?? $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->with('drupal_purview')->willReturn($loggerChannel);

    // Translation service (StringTranslationTrait will use it).
    $translation = $translation ?? $this->createMock(TranslationInterface::class);
    // Map translate() to return the string.
    $translation->method('translate')->willReturnCallback(function ($string, array $args = []) {
      // Lightweight interpolation for test stability.
      return strtr($string, array_combine(array_keys($args), array_values($args)));
    });

    return new PurviewGovernanceDomainApiClient(
      $configFactory,
      $client,
      $loggerFactory,
      $auth,
      $cache,
      $translation,
    );
  }

  /**
   * Small helper to build a Guzzle client backed by MockHandler.
   *
   * @param \Psr\Http\Message\ResponseInterface[] $responses
   *   Queue of responses to return.
   *
   * @return \GuzzleHttp\ClientInterface
   *   A client that will return the queued responses in order.
   */
  private function makeClient(array $responses): ClientInterface {
    $mock = new MockHandler($responses);
    $stack = HandlerStack::create($mock);
    return new Client(['handler' => $stack]);
  }

  /**
   * Build a config factory that returns just the settings this client reads.
   *
   * @param string $guid
   *   Purview account GUID to inject.
   * @param string $apiVersion
   *   API version to return.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   A factory whose ->get('drupal_purview.settings')->get()
   *   serves our values.
   */
  private function configFactoryWithGuid(string $guid, string $apiVersion = '2023-10-01-preview'): ConfigFactoryInterface {
    $config = $this->createMock(Config::class);
    $config->method('get')
      ->willReturnMap([
        ['purview_account_guid', $guid],
        ['api_version', $apiVersion],
      ]);

    $factory = $this->createMock(ConfigFactoryInterface::class);
    $factory->method('get')->with('drupal_purview.settings')->willReturn($config);
    return $factory;
  }

}
