/**
 * @file
 * Handles toggle interactions for governance domain items.
 *
 * - When a toggle button is clicked, it switches the 'open' class on the item
 *   and updates 'aria-expanded' for accessibility.
 * - When a domain link is clicked, it removes the 'active' class from all
 *   domain items and adds it to the clicked item's wrapper.
 */
(function (Drupal, once) {
  Drupal.behaviors.governanceDomainsToggle = {
    attach(context) {
      // Toggle open class.
      once('governance-domains-toggle', '.toggle-children', context).forEach(button => {
        button.addEventListener('click', () => {
          const expanded = button.getAttribute('aria-expanded') === 'true';
          button.setAttribute('aria-expanded', String(!expanded));

          const item = button.closest('.governance-domain-item');
          item.classList.toggle('open', !expanded);
        });
      });

      // Toggle active class.
      once('governance-domains-active', '.domain-link', context).forEach(link => {
        link.addEventListener('click', (e) => {
          // Remove 'active' from all domain link wrappers.
          document.querySelectorAll('.governance-domain-item.active').forEach(activeLink => {
            activeLink.classList.remove('active');
          });

          // Add 'active' to the clicked link wrapper.
          const item = link.closest('.governance-domain-item');
          if (item) {
            item.classList.add('active');

            // Force the item open.
            item.classList.add('open');

            // Also update the toggle button's aria-expanded attribute.
            const toggleButton = item.querySelector('.toggle-children');
            if (toggleButton) {
              toggleButton.setAttribute('aria-expanded', 'true');
            }
          }
        });
      });
    }
  };
})(Drupal, once);
