<?php

namespace Drupal\purview_unified_catalog_ui\Breadcrumb;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Url;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;

/**
 * Builds custom breadcrumbs for the glossary term metadata page.
 *
 * Replaces the default breadcrumb with a custom trail:
 *   Home > Business glossary > [Term Title]
 * The final breadcrumb is rendered as plain text (not a link).
 */
class GlossaryTermBreadcrumbBuilder implements BreadcrumbBuilderInterface {

  /**
   * The Purview Governance Domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $apiClient;

  /**
   * Constructs a new GlossaryTermBreadcrumbBuilder.
   *
   * @param \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient $apiClient
   *   The Purview API client for retrieving term metadata.
   */
  public function __construct(PurviewGovernanceDomainApiClient $apiClient) {
    $this->apiClient = $apiClient;
  }

  /**
   * Determines whether this breadcrumb builder applies to the current route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   *
   * @return bool
   *   TRUE if this builder should handle the route, FALSE otherwise.
   */
  public function applies(RouteMatchInterface $route_match): bool {
    return $route_match->getRouteName() === 'purview_unified_catalog_ui.glossary_metadata_page';
  }

  /**
   * Builds the breadcrumb for the glossary term metadata page.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   *
   * @return \Drupal\Core\Breadcrumb\Breadcrumb
   *   The constructed breadcrumb object.
   */
  public function build(RouteMatchInterface $route_match): Breadcrumb {
    $breadcrumb = new Breadcrumb();

    $breadcrumb->addLink(Link::createFromRoute(t('Home'), '<front>'));
    $breadcrumb->addLink(Link::createFromRoute(t('Business glossary'), 'purview_unified_catalog_ui.glossary_search'));

    // Get GUID and lookup term.
    $guid = $route_match->getRawParameter('guid');
    if ($guid) {
      $term = $this->apiClient->getGlossaryTermMetadata($guid);
      if (!empty($term['name'])) {
        $breadcrumb->addLink(Link::fromTextAndUrl($term['name'], Url::fromRoute('<none>')));
        $breadcrumb->addCacheTags(["purview_term:$guid"]);
      }
    }

    // Cache context for the URL to prevent bleeding across routes.
    $breadcrumb->addCacheContexts(['url']);

    return $breadcrumb;
  }

}
