<?php

namespace Drupal\purview_unified_catalog_ui\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenOffCanvasDialogCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\drupal_purview\Service\GraphApiClient;
use Drupal\drupal_purview\Utility\PurviewHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying product metadata in an off-canvas panel.
 *
 * This controller retrieves metadata for a given product,
 * renders it using custom templates, and returns it
 * in an off-canvas dialog with tabs for details and data assets.
 */
class UnifiedProductMetadataController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionListModule;

  /**
   * The governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $governanceDomainApiClient;

  /**
   * The Graph API client.
   *
   * @var \Drupal\drupal_purview\Service\GraphApiClient
   */
  protected GraphApiClient $graphApiClient;

  /**
   * The Purview helper service.
   *
   * @var \Drupal\drupal_purview\Utility\PurviewHelper
   */
  protected PurviewHelper $purviewHelper;

  /**
   * Constructs the UnifiedProductMetadataController.
   */
  public function __construct(
    RendererInterface $renderer,
    ModuleExtensionList $extensionListModule,
    PurviewGovernanceDomainApiClient $governanceDomainApiClient,
    GraphApiClient $graphApiClient,
    PurviewHelper $purviewHelper,
    ConfigFactoryInterface $configFactory,
  ) {
    $this->renderer = $renderer;
    $this->extensionListModule = $extensionListModule;
    $this->governanceDomainApiClient = $governanceDomainApiClient;
    $this->graphApiClient = $graphApiClient;
    $this->purviewHelper = $purviewHelper;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('renderer'),
      $container->get('extension.list.module'),
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('drupal_purview.graph_api_client'),
      $container->get('drupal_purview.helper'),
      $container->get('config.factory')
    );
  }

  /**
   * Opens the product metadata in an off-canvas panel.
   *
   * Retrieves the product title for the page title and builds the content
   * render array using the shared metadata builder.
   *
   * @param string $guid
   *   The GUID of the product.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response containing the off-canvas command.
   */
  public function openOffcanvas(string $guid): AjaxResponse {
    $title = $this->governanceDomainApiClient->getDataProductMetadata($guid)['name'];
    $build = [
      '#type' => 'off_canvas',
      '#title' => $title,
      'content' => $this->buildProductMetadataRenderArray($guid, TRUE),
    ];
    $response = new AjaxResponse();
    $response->addCommand(new OpenOffCanvasDialogCommand(
      $title,
      $build,
      ['dialogClass' => 'product-metadata-offcanvas']
    ));
    return $response;
  }

  /**
   * Returns a full-page render array for a data product's metadata.
   *
   * Retrieves the product title for the page title and builds the content
   * render array using the shared metadata builder.
   *
   * @param string $guid
   *   The GUID of the data product.
   *
   * @return array
   *   A render array for displaying the metadata as a full page.
   */
  public function page(string $guid): array {
    return [
      '#title' => $this->governanceDomainApiClient->getDataProductMetadata($guid)['name'],
      'content' => $this->buildProductMetadataRenderArray($guid, TRUE),
    ];
  }

  /**
   * Builds the render array for displaying a data product's metadata.
   *
   * This method prepares the metadata display including owners, domain,
   * score, authoring info, and related data assets. The result is used
   * for both off-canvas and full-page rendering.
   *
   * @param string $guid
   *   The GUID of the data product.
   *
   * @return array
   *   A render array representing the full metadata view, including tabs.
   */
  protected function buildProductMetadataRenderArray(string $guid): array {
    $module_path = $this->extensionListModule->getPath('purview_unified_catalog_ui');
    $product = $this->governanceDomainApiClient->getDataProductMetadata($guid);
    $product_id = $product['id'];
    $title = $product['name'];

    // Owners.
    $owner_output = '';
    if (!empty($product['contacts']['owner'])) {
      $owner_output = $this->purviewHelper->renderOwnersSmall($product['contacts']['owner']);
    }

    // Score.
    $score = $this->governanceDomainApiClient->getAverageDataQualityScoreByKey('dataProductId', $product_id);
    $label = $score ? $this->governanceDomainApiClient->getDataQualityLabel($score) : $this->t('No score available');
    $score_array = [
      'score' => $score ?: NULL,
      'desc' => $label,
      'class' => $score ? 'score--' . strtolower($label) : 'muted',
    ];

    // Domain.
    $domain = $this->governanceDomainApiClient->getGovernanceDomainMetadata($product['domain']);
    $first_letter = strtoupper($domain['name'][0]);
    $colors = PurviewHelper::generateLetterColors($first_letter);
    $domain_info = [
      'name' => $domain['name'],
      'first_letter' => $first_letter,
      'background_color' => $colors['background'],
      'border_color' => $colors['border'],
    ];

    // Authoring info.
    $timezone = $this->configFactory->get('system.date')->get('timezone.default');
    $create_time = (new \DateTime($product['systemData']['createdAt'], new \DateTimeZone('UTC')))
      ->setTimezone(new \DateTimeZone($timezone))
      ->format('n/j/Y, g:i A');

    $update_time = (new \DateTime($product['systemData']['lastModifiedAt'], new \DateTimeZone('UTC')))
      ->setTimezone(new \DateTimeZone($timezone))
      ->format('n/j/Y, g:i A');

    $created_by = $this->graphApiClient->resolveUserNameByGuid($product['systemData']['createdBy']);
    $updated_by = $this->graphApiClient->resolveUserNameByGuid($product['systemData']['lastModifiedBy']);

    $authoring_info = [
      'create_time' => $create_time,
      'update_time' => $update_time,
      'created_by' => $created_by,
      'updated_by' => $updated_by,
    ];

    // Data assets.
    $config = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $exclude_types = $config->get('data_asset_exclude');
    $asset_map = $config->get('data_asset_mapping');
    $data_assets = $this->governanceDomainApiClient->getDataAssetsForDataProduct($product_id);
    $assets = $this->governanceDomainApiClient->getDataAssetMetadataByIds($data_assets['value']);
    $assets = $this->normalizeAndFilterDataAssets($assets, $exclude_types, $asset_map);

    // Data assets description.
    $data_assets_description = $config->get('data_assets_description.value') ?? '';

    // Audience.
    $audience = $product['audience'] ?? [];
    if (!empty($audience) && is_array($audience)) {
      $audience_array = array_map(function ($item) {
        // Add a space before each uppercase letter (except the first),
        // then lowercase and ucfirst.
        return ucfirst(strtolower(preg_replace('/(?<!^)([A-Z])/', ' $1', $item)));
      }, $audience);
    }
    else {
      $audience_array = ['--'];
    }

    // Associated terms.
    $terms_id_array = $this->governanceDomainApiClient->getGlossaryTermsForDataProduct($product_id);

    if (!empty($terms_id_array['value'])) {
      $term_id_array = [];

      foreach ($terms_id_array['value'] as $term) {
        $term_id_array[] = $term['entityId'];
      }

      $terms = $this->governanceDomainApiClient->searchGlossaryTerms('', '', $term_id_array);
    }

    $term_array = [];
    if (!empty($terms['value'])) {
      foreach ($terms['value'] as $term_data) {
        $term_array[$term_data['id']] = $term_data['name'];
      }
    }

    // Tabs navigation.
    $links_markup = Markup::create('
      <nav class="tabs product-tabs-links" aria-label="Tabs">
        <h2 class="visually-hidden">' . $this->t('Data product tabs') . '</h2>
        <ul role="menu" class="tabs--primary nav nav-tabs">
          <li role="presentation">
            <a href="#" class="product-tab-link active" data-target="general">' . $this->t('Details') . '</a>
          </li>
          <li role="presentation">
            <a href="#" class="product-tab-link" data-target="assets">' . $this->t('Data assets') . '</a>
          </li>
        </ul>
      </nav>
    ');

    // Canonical URL.
    $canonical_url = Url::fromRoute('purview_unified_catalog_ui.product_metadata_page', ['guid' => $guid], ['absolute' => TRUE])->toString();

    // Messages.
    $messages_build = [
      '#type' => 'status_messages',
    ];
    $messages_rendered = $this->renderer->render($messages_build);
    $messages = '<div data-drupal-messages>' . $messages_rendered . '</div>';

    return [
      '#attached' => [
        'library' => ['purview_unified_catalog_ui/unified_product_metadata_offcanvas'],
      ],
      'messages' => [
        '#markup' => $messages,
      ],
      'tabs_links' => [
        '#markup' => $links_markup,
      ],
      'tabs_content' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['product-tabs-content']],
        'general_tab' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'general-tab'],
          'content' => [
            '#theme' => 'unified_catalog_product_metadata',
            '#data' => $product,
            '#name' => $title,
            '#status' => $product['status'],
            '#score' => $score_array,
            '#owners' => $owner_output,
            '#description' => $product['description'],
            '#use' => $product['businessUse'],
            '#domain' => $domain_info,
            '#authoring_info' => $authoring_info,
            '#module_path' => $module_path,
            '#audience' => $audience_array,
            '#terms' => $term_array,
            '#canonical_url' => $canonical_url,
          ],
        ],
        'assets_tab' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'assets-tab'],
          'content' => [
            '#theme' => 'unified_catalog_product_assets',
            '#data_assets' => $assets,
            '#description' => $data_assets_description,
            '#module_path' => $module_path,
          ],
        ],
      ],
    ];
  }

  /**
   * Normalizes asset types using a textarea-provided map, then excludes types.
   *
   * Notes:
   *  - Exclusions are matched against machine keys (e.g., "powerbi_report"),
   *    not the human labels (e.g., "PowerBI Report").
   *  - Lines in $asset_map without a "|" are ignored. Blank lines are ignored.
   *  - Whitespace around keys/labels in $asset_map are trimmed.
   *  - Original array keys from $assets are preserved.
   *
   * @param array $assets
   *   A list of data assets. Each asset may contain
   *   $asset['source']['assetType'] as a machine key string.
   * @param string|null $exclude_types
   *   Comma-separated list of machine type keys to exclude (case-insensitive),
   *   e.g., "powerbi_dataset,file".
   * @param string|null $asset_map
   *   Textarea content where each line is "machine_key|Human Label".
   *   Lines without a "|" are ignored. Example:
   *     powerbi_dataset|PowerBI Dataset
   *     powerbi_report|PowerBI Report.
   *
   * @return array
   *   The list of assets after:
   *     - Mapping known machine keys to human labels, and
   *     - Removing assets whose machine key is in $exclude_types.
   *   Original array keys are preserved.
   */
  public function normalizeAndFilterDataAssets(array $assets, ?string $exclude_types, ?string $asset_map): array {
    // Build the exclusion list (lowercased, trimmed).
    $exclude_list = [];
    if (!empty($exclude_types)) {
      $exclude_list = array_map('strtolower', array_map('trim', explode(',', $exclude_types)));
    }

    // Parse textarea mapping into associative array: machine_key => label.
    $type_map = [];
    if (!empty($asset_map)) {
      // Split on any newline type.
      $lines = preg_split('/\r\n|\r|\n/', $asset_map);
      foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || strpos($line, '|') === FALSE) {
          // Ignore blank or malformed lines.
          continue;
        }
        // Split on the first "|" only, to allow labels that might contain "|".
        [$raw_key, $raw_label] = array_pad(explode('|', $line, 2), 2, '');
        $key = strtolower(trim($raw_key));
        $label = trim($raw_label);

        if ($key !== '' && $label !== '') {
          $type_map[$key] = $label;
        }
      }
    }

    $result = [];
    foreach ($assets as $k => $asset) {
      // Read the raw machine type.
      $raw_type = $asset['source']['assetType'] ?? NULL;
      $raw_type_lower = is_string($raw_type) ? strtolower($raw_type) : '';

      // Exclude if the machine key is listed.
      if ($raw_type_lower !== '' && in_array($raw_type_lower, $exclude_list, TRUE)) {
        continue;
      }

      // Always try to map to human label if mapping exists.
      if ($raw_type_lower !== '' && isset($type_map[$raw_type_lower])) {
        $asset['source']['assetType'] = $type_map[$raw_type_lower];
      }

      // Preserve original array keys.
      $result[$k] = $asset;
    }

    return $result;
  }

}
