<?php

namespace Drupal\purview_unified_catalog_ui\Utility;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Provides helper utilities for the drupal_purview module.
 */
class UnifiedCatalogHelper {

  /**
   * The config factory interface.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new PurviewHelper instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory interface.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Extract managed attributes from a term or product, filtered by config.
   *
   * The config 'purview_unified_catalog_ui.settings:attribute_filter' should be
   * newline-separated tokens like "Group|Field" or "Group.Field".
   *
   * @param array $object
   *   The glossary term or data product.
   *
   * @return string[]
   *   Flat array of values (e.g., ["Attribute Value"]).
   */
  public function extractManagedAttributeValues(array $object): array {
    $values = [];

    // Nothing to do if no attributes.
    $attrs = $object['managedAttributes'] ?? NULL;
    if (!is_array($attrs)) {
      return $values;
    }

    // Build allowed set from config.
    $config = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $filter_raw = (string) ($config->get('attribute_filter') ?? '');
    $allowed = [];

    if ($filter_raw !== '') {
      foreach (preg_split('/\R/', $filter_raw) as $line) {
        $line = trim($line);
        if ($line === '') {
          continue;
        }
        // Accept either "Group|Field" or "Group.Field".
        $norm = strtolower(str_replace('.', '|', $line));
        $allowed[$norm] = TRUE;
      }
    }

    foreach ($attrs as $attr) {
      $name = (string) ($attr['name'] ?? '');
      if ($name === '') {
        continue;
      }

      // Normalize the attribute's name: "Group.Field" -> "group|field".
      // We only split on the FIRST dot, in case field labels contain dots.
      if (strpos($name, '.') !== FALSE) {
        [$group, $field] = explode('.', $name, 2);
        $pair = strtolower(trim($group) . '|' . trim($field));
      }
      else {
        // If no dot is present, treat the whole as the field.
        $pair = strtolower('|' . trim($name));
      }

      // If a filter is defined, require a match.
      if (!empty($allowed) && empty($allowed[$pair])) {
        continue;
      }

      // Pull and normalize the value(s).
      if (!array_key_exists('value', $attr)) {
        continue;
      }
      $raw = trim((string) $attr['value']);
      if ($raw === '') {
        continue;
      }

      $decoded = json_decode($raw, TRUE);
      if (json_last_error() === JSON_ERROR_NONE) {
        if (is_array($decoded)) {
          foreach ($decoded as $item) {
            $v = trim((string) $item);
            if ($v !== '') {
              $values[] = $v;
            }
          }
        }
        else {
          $v = trim((string) $decoded);
          if ($v !== '') {
            $values[] = $v;
          }
        }
      }
      else {
        // Non-JSON: use the raw string.
        $values[] = $raw;
      }
    }

    return $values;
  }

}
