<?php

namespace Drupal\drupal_purview\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller to flush the Purview custom cache bin.
 */
class FlushCacheController extends ControllerBase {

  /**
   * The Purview cache bin.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $purviewCache;

  /**
   * The current request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs the FlushCacheController.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $purviewCache
   *   The Purview cache backend.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   */
  public function __construct(CacheBackendInterface $purviewCache, RequestStack $requestStack) {
    $this->purviewCache = $purviewCache;
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('cache.purview'),
      $container->get('request_stack')
    );
  }

  /**
   * Clears the Purview cache and redirects back to the referring page.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   The redirect response.
   */
  public function purviewCache(): RedirectResponse {
    $this->purviewCache->deleteAll();
    $this->messenger()->addStatus($this->t('Purview cache cleared.'));

    $referer = $this->requestStack->getCurrentRequest()->headers->get('referer');
    return new RedirectResponse($referer ?? '/admin/config/development/performance');
  }

}
