<?php

namespace Drupal\drupal_purview;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\drupal_purview\Service\GraphApiClient;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;

/**
 * Service to warm up cache for common Purview API calls.
 */
class PurviewCacheWarmer {

  /**
   * The Purview API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $apiClient;

  /**
   * The cache backend used for Purview API data.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The Microsoft Graph API client service.
   *
   * Used to retrieve user objects by GUID from Azure AD.
   *
   * @var \Drupal\drupal_purview\Service\GraphApiClient
   */
  protected GraphApiClient $graphClient;

  /**
   * Constructs the cache warmer.
   *
   * @param \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient $apiClient
   *   The Purview API client.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\drupal_purview\Service\GraphApiClient $graphClient
   *   The graph API client.
   */
  public function __construct(
    PurviewGovernanceDomainApiClient $apiClient,
    CacheBackendInterface $cache,
    GraphApiClient $graphClient,
  ) {
    $this->apiClient = $apiClient;
    $this->cache = $cache;
    $this->graphClient = $graphClient;
  }

  /**
   * Clears and pre-populates the Purview cache with commonly used API calls.
   *
   * This warms the cache for:
   * - Governance domains list
   * - Metadata and health status for each domain
   * - Count of data products and glossary terms per domain
   * - Facet values for data products and glossary terms
   * - Data quality report.
   */
  public function warm(): void {
    // Clear the purview cache bin before warming.
    $this->cache->deleteAll();

    // Domain list.
    $domains = $this->apiClient->getGovernanceDomains(TRUE) ?? [];

    foreach ($domains as $domain) {
      $id = $domain['id'] ?? NULL;
      if (!$id) {
        continue;
      }

      // Warm metadata and metrics for each domain.
      $this->apiClient->getGovernanceDomainMetadata($id, TRUE);
      $this->apiClient->getGovernanceDomainHealthActionsCount($id, TRUE);
      $this->apiClient->getDataProductCountForDomain($id);
      $this->apiClient->getGlossaryTermCountForDomain($id);
    }

    // Warm data quality report.
    $this->apiClient->getDataQualityReport(TRUE);

    // Warm Graph API cache for all owners.
    $owner_guids = [];

    // Collect unique owner GUIDs from both facets.
    foreach (['dataproducts', 'terms'] as $resource) {
      $facets = $this->apiClient->getFacetValues($resource, ['owner'], TRUE);
      if (!empty($facets['owner'])) {
        foreach ($facets['owner'] as $owner) {
          if (!empty($owner['value'])) {
            $owner_guids[$owner['value']] = TRUE;
          }
        }
      }
    }

    // Load user objects to warm the Graph API cache.
    foreach (array_keys($owner_guids) as $guid) {
      $this->graphClient->getUserObjectFromGuid($guid);
    }
  }

}
