<?php

declare(strict_types=1);

namespace Drupal\Tests\drupal_purview\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\drupal_purview\Service\PurviewAuthenticationService;
use Drupal\drupal_purview\Testing\FixtureLoaderTrait;
use Drupal\drupal_purview\Testing\HttpClientFactory;
use GuzzleHttp\Psr7\Response;

/**
 * Kernel test to verify caching behavior in the Purview API client.
 *
 * Ensures that API results are cached after the first request and subsequent
 * calls reuse the cache rather than hitting the HTTP client again.
 *
 * @group drupal_purview
 * @covers \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
 */
final class PurviewCachingKernelTest extends KernelTestBase {
  use FixtureLoaderTrait;

  /**
   * Minimal module set required to bootstrap the test container.
   *
   * @var string[]
   */
  protected static $modules = [
    'system',
    'user',
    'key',
    'drupal_purview',
  ];

  /**
   * Set up configuration and service stubs before each test.
   *
   * - Adds a fake Purview REST endpoint.
   * - Stubs the authentication service to always return a static token.
   *   This avoids requiring the Key module or real credentials.
   */
  protected function setUp(): void {
    parent::setUp();
    // Minimal config to satisfy the client.
    $this->config('drupal_purview.settings')
      ->set('rest_endpoint', 'https://example.purview')
      ->save();

    // Stub the auth service so the API client always receives a token.
    // This avoids a hard dependency on the Key module/real credentials
    // during tests.
    $auth = $this->getMockBuilder(PurviewAuthenticationService::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getAccessToken'])
      ->getMock();
    $auth->method('getAccessToken')->willReturn('test-token');
    $this->container->set('drupal_purview.auth_service', $auth);
  }

  /**
   * Tests that cached results are reused between API calls.
   *
   * Steps:
   * 1. Call the API once using a working HTTP client (fixture JSON).
   * 2. Replace the HTTP client with one that always fails.
   * 3. Call the API again without bypassing the cache.
   * 4. Assert that the second call still returns the cached data.
   */
  public function testCacheIsUsedBetweenCalls(): void {
    $client = HttpClientFactory::fromResponses([
      new Response(200, ['Content-Type' => 'application/json'], $this->loadFixture('atlas_typedefs_ok.json')),
    ]);

    // Override the global http_client service with our mocked client.
    $this->container->set('http_client', $client);

    /** @var \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient $sut */
    $sut = $this->container->get('drupal_purview.api_client');

    // First call should hit HTTP and populate cache.
    $first = $sut->getManagedAttributeDefinitions(TRUE);
    $this->assertIsArray($first, 'First call returns an array of definitions.');
    $this->assertArrayHasKey('MyGroup.MyField', $first, 'Fixture contains expected field.');

    // Swap in a failing client to prove we don't hit HTTP again.
    $failing = HttpClientFactory::fromResponses([
      new Response(500, ['Content-Type' => 'application/json'], '{"error":"boom"}'),
    ]);
    $this->container->set('http_client', $failing);

    // Second call without bypass should read from cache and still succeed.
    $second = $sut->getManagedAttributeDefinitions(FALSE);
    $this->assertEquals($first, $second, 'Second call returned cached data.');
  }

}
