/**
 * @file
 * D3.js-based DFD diagram generator for Drupal entities.
 */

(function ($, Drupal) {
  'use strict';

  // DFD Diagram class
  class DFDDiagram {
    constructor(container, data) {
      this.container = d3.select(container);
      this.data = data;
      this.width = 1400;
      this.height = 800;
      this.nodeWidth = 200;
      this.nodeHeight = 120;
      this.margin = { top: 20, right: 20, bottom: 20, left: 20 };
      
      this.svg = null;
      this.nodes = [];
      this.links = [];
      
      this.init();
    }

    init() {
      // Clear existing content
      this.container.html('');
      
      // Create SVG
      this.svg = this.container
        .append('svg')
        .attr('width', this.width)
        .attr('height', this.height)
        .style('background', '#1a1a1a')
        .style('border', '1px solid #333');

      // Add zoom behavior
      const zoom = d3.zoom()
        .scaleExtent([0.1, 3])
        .on('zoom', (event) => {
          this.svg.select('.main-group').attr('transform', event.transform);
        });

      this.svg.call(zoom);

      // Main group for zoom/pan
      this.mainGroup = this.svg.append('g').attr('class', 'main-group');

      // Process data and create diagram
      this.processData();
      this.createDiagram();
    }

    processData() {
      let nodeId = 0;
      this.nodes = [];
      this.links = [];

      // Create nodes for content types
      Object.entries(this.data.bundles || {}).forEach(([bundleId, bundle]) => {
        const node = {
          id: bundleId,
          displayId: nodeId++,
          type: 'content_type',
          label: bundle.label || bundle.bundle,
          entityType: 'CT',
          bundle: bundle.bundle,
          fields: bundle.fields || {},
          x: 0,
          y: 0
        };
        this.nodes.push(node);
      });

      // Create nodes for paragraphs
      Object.entries(this.data.paragraphs || {}).forEach(([bundleId, bundle]) => {
        const node = {
          id: bundleId,
          displayId: nodeId++,
          type: 'paragraph',
          label: bundle.label || bundle.bundle,
          entityType: 'Para',
          bundle: bundle.bundle,
          fields: bundle.fields || {},
          x: 0,
          y: 0
        };
        this.nodes.push(node);
      });

      // Create nodes for taxonomies
      Object.entries(this.data.taxonomies || {}).forEach(([bundleId, bundle]) => {
        const node = {
          id: bundleId,
          displayId: nodeId++,
          type: 'taxonomy',
          label: bundle.label || bundle.bundle,
          entityType: 'Voc',
          bundle: bundle.bundle,
          x: 0,
          y: 0
        };
        this.nodes.push(node);
      });

      // Create nodes for media types
      Object.entries(this.data.media || {}).forEach(([bundleId, bundle]) => {
        const node = {
          id: bundleId,
          displayId: nodeId++,
          type: 'media',
          label: bundle.label || bundle.bundle,
          entityType: 'Media',
          bundle: bundle.bundle,
          x: 0,
          y: 0
        };
        this.nodes.push(node);
      });

      // Create nodes for webforms
      Object.entries(this.data.webforms || {}).forEach(([bundleId, bundle]) => {
        const node = {
          id: bundleId,
          displayId: nodeId++,
          type: 'webform',
          label: bundle.label || bundle.bundle,
          entityType: 'Form',
          bundle: bundle.bundle || bundle.label || bundleId,
          fields: bundle.fields || {},
          x: 0,
          y: 0
        };
        this.nodes.push(node);
      });

      // Create links from relationships
      (this.data.relationships || []).forEach((rel) => {
        const sourceNode = this.nodes.find(n => n.id === rel.source);
        
        if (sourceNode && rel.target_bundles) {
          rel.target_bundles.forEach((targetBundle) => {
            const targetId = rel.target_entity_type + '.' + targetBundle;
            const targetNode = this.nodes.find(n => n.id === targetId);
            
            if (targetNode) {
              this.links.push({
                source: sourceNode,
                target: targetNode,
                label: rel.field,
                type: rel.type || 'entity_reference'
              });
            }
          });
        }
      });

      // Calculate layout
      this.calculateLayout();
    }

    calculateLayout() {
      const typeOrder = ['content_type', 'paragraph', 'taxonomy', 'media', 'webform'];
      const grouped = typeOrder
        .map(type => ({
          type,
          nodes: this.nodes.filter(node => node.type === type)
        }))
        .filter(group => group.nodes.length);

      if (!grouped.length) {
        grouped.push({ type: 'all', nodes: this.nodes });
      }

      const columnCount = grouped.length;
      const maxRows = Math.max(...grouped.map(group => group.nodes.length));

      const minWidth = columnCount * (this.nodeWidth + 120) + this.margin.left + this.margin.right;
      if (minWidth > this.width) {
        this.width = minWidth;
        if (this.svg) {
          this.svg.attr('width', this.width);
        }
      }

      const minHeight = maxRows * (this.nodeHeight + 40) + this.margin.top + this.margin.bottom;
      if (minHeight > this.height) {
        this.height = minHeight;
        if (this.svg) {
          this.svg.attr('height', this.height);
        }
      }

      const availableWidth = this.width - this.margin.left - this.margin.right;
      const columnSpacing = columnCount > 0 ? availableWidth / columnCount : availableWidth;
      const availableHeight = this.height - this.margin.top - this.margin.bottom;

      grouped.forEach((group, columnIndex) => {
        // Keep bundle order predictable for each column
        group.nodes.sort((a, b) => {
          const labelA = (a.label || a.bundle || '').toLowerCase();
          const labelB = (b.label || b.bundle || '').toLowerCase();
          return labelA.localeCompare(labelB);
        });

        const rowCount = group.nodes.length;
        if (!rowCount) {
          return;
        }

        const rowSpacing = Math.max(this.nodeHeight + 40, availableHeight / rowCount);
        const columnCenter = this.margin.left + columnIndex * columnSpacing + columnSpacing / 2;
        const totalColumnHeight = rowSpacing * rowCount;
        const startY = this.margin.top + Math.max(0, (availableHeight - totalColumnHeight) / 2);

        group.nodes.forEach((node, rowIndex) => {
          node.x = columnCenter;
          node.y = startY + rowIndex * rowSpacing + this.nodeHeight / 2;
        });
      });
    }

    createDiagram() {
      // Create links first (so they appear behind nodes)
      this.createLinks();
      
      // Create nodes
      this.createNodes();
      
      // Add title
      this.mainGroup.append('text')
        .attr('x', this.width / 2)
        .attr('y', 30)
        .attr('text-anchor', 'middle')
        .style('fill', '#ffffff')
        .style('font-size', '20px')
        .style('font-weight', 'bold')
        .text('Project DFD - Entity Relationships');
    }

    createNodes() {
      const nodeGroup = this.mainGroup
        .selectAll('.node')
        .data(this.nodes)
        .enter()
        .append('g')
        .attr('class', 'node')
        .attr('transform', d => `translate(${d.x - this.nodeWidth/2}, ${d.y - this.nodeHeight/2})`);

      // Node background
      nodeGroup
        .append('rect')
        .attr('width', this.nodeWidth)
        .attr('height', this.nodeHeight)
        .attr('rx', 5)
        .style('fill', d => this.getNodeColor(d.type))
        .style('stroke', d => this.getNodeBorderColor(d.type))
        .style('stroke-width', 2);

      // Node header
      nodeGroup
        .append('rect')
        .attr('width', this.nodeWidth)
        .attr('height', 25)
        .attr('rx', 5)
        .style('fill', d => this.getNodeHeaderColor(d.type))
        .style('stroke', 'none');

      // Header rounding fix
      nodeGroup
        .append('rect')
        .attr('y', 20)
        .attr('width', this.nodeWidth)
        .attr('height', 5)
        .style('fill', d => this.getNodeHeaderColor(d.type))
        .style('stroke', 'none');

      // Entity type label
      nodeGroup
        .append('text')
        .attr('x', 8)
        .attr('y', 18)
        .style('fill', '#ffffff')
        .style('font-size', '12px')
        .style('font-weight', 'bold')
        .text(d => d.entityType);

      // Bundle name
      nodeGroup
        .append('text')
        .attr('x', 60)
        .attr('y', 18)
        .style('fill', '#ffffff')
        .style('font-size', '12px')
        .text(d => d.bundle || d.label || '');

      // Fields
      nodeGroup.each(function(d) {
        const node = d3.select(this);
        let yOffset = 40;
        
        Object.entries(d.fields || {}).slice(0, 6).forEach(([fieldName, field]) => {
          // Field icon
          node.append('text')
            .attr('x', 8)
            .attr('y', yOffset)
            .style('fill', '#4a9eff')
            .style('font-size', '10px')
            .text('+');

          // Field name
          node.append('text')
            .attr('x', 20)
            .attr('y', yOffset)
            .style('fill', '#e6e6e6')
            .style('font-size', '10px')
            .text(fieldName.length > 25 ? fieldName.substring(0, 25) + '...' : fieldName);

          // Field type
          node.append('text')
            .attr('x', this.nodeWidth - 60)
            .attr('y', yOffset)
            .style('fill', '#999')
            .style('font-size', '9px')
            .text(`(${field.type || 'field'})`);

          yOffset += 12;
        });

        // Show "..." if there are more fields
        const fieldCount = Object.keys(d.fields || {}).length;
        if (fieldCount > 6) {
          node.append('text')
            .attr('x', 8)
            .attr('y', yOffset)
            .style('fill', '#666')
            .style('font-size', '10px')
            .text(`... and ${fieldCount - 6} more fields`);
        }
      }.bind(this));
    }

    createLinks() {
      const linkGroup = this.mainGroup
        .selectAll('.link')
        .data(this.links)
        .enter()
        .append('g')
        .attr('class', 'link');

      // Link lines
      linkGroup
        .append('line')
        .attr('x1', d => d.source.x)
        .attr('y1', d => d.source.y)
        .attr('x2', d => d.target.x)
        .attr('y2', d => d.target.y)
        .style('stroke', '#ff6b9d')
        .style('stroke-width', 2)
        .style('opacity', 0.8);

      // Arrow heads
      this.mainGroup.append('defs').append('marker')
        .attr('id', 'arrowhead')
        .attr('viewBox', '0 -5 10 10')
        .attr('refX', 8)
        .attr('refY', 0)
        .attr('markerWidth', 8)
        .attr('markerHeight', 8)
        .attr('orient', 'auto')
        .append('path')
        .attr('d', 'M0,-5L10,0L0,5')
        .style('fill', '#ff6b9d');

      linkGroup.select('line')
        .attr('marker-end', 'url(#arrowhead)');

      // Link labels
      linkGroup
        .append('text')
        .attr('x', d => (d.source.x + d.target.x) / 2)
        .attr('y', d => (d.source.y + d.target.y) / 2 - 5)
        .attr('text-anchor', 'middle')
        .style('fill', '#ffffff')
        .style('font-size', '10px')
        .style('background', '#000')
        .text(d => d.label);
    }

    getNodeColor(type) {
      const colors = {
        content_type: '#2d3436',
        paragraph: '#2d3436',
        taxonomy: '#2d3436',
        media: '#2d3436',
        webform: '#2d3436'
      };
      return colors[type] || '#2d3436';
    }

    getNodeBorderColor(type) {
      const colors = {
        content_type: '#4a90e2',
        paragraph: '#7b68ee',
        taxonomy: '#ff6b6b',
        media: '#26d0ce',
        webform: '#ffa500'
      };
      return colors[type] || '#666';
    }

    getNodeHeaderColor(type) {
      const colors = {
        content_type: '#4a90e2',
        paragraph: '#7b68ee', 
        taxonomy: '#ff6b6b',
        media: '#26d0ce',
        webform: '#ffa500'
      };
      return colors[type] || '#666';
    }

    // Export methods
    exportAsSVG() {
      const svgData = this.container.select('svg').node().outerHTML;
      const blob = new Blob([svgData], { type: 'image/svg+xml' });
      const url = URL.createObjectURL(blob);
      
      const link = document.createElement('a');
      link.download = 'dfd-diagram.svg';
      link.href = url;
      link.click();
      
      URL.revokeObjectURL(url);
    }

    exportAsPNG() {
      const svgElement = this.container.select('svg').node();
      const svgData = new XMLSerializer().serializeToString(svgElement);
      
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = new Image();
      
      canvas.width = this.width * 2; // 2x for better quality
      canvas.height = this.height * 2;
      ctx.scale(2, 2);
      
      img.onload = () => {
        ctx.fillStyle = '#1a1a1a';
        ctx.fillRect(0, 0, this.width, this.height);
        ctx.drawImage(img, 0, 0, this.width, this.height);
        
        canvas.toBlob(blob => {
          const url = URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.download = 'dfd-diagram.png';
          link.href = url;
          link.click();
          URL.revokeObjectURL(url);
        }, 'image/png');
      };
      
      img.src = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
    }
  }

  // Global functions for export (defined immediately)
  window.exportDiagramAsSVG = function() {
    console.log('exportDiagramAsSVG called', window.dfdDiagram);
    if (window.dfdDiagram) {
      window.dfdDiagram.exportAsSVG();
    } else {
      alert('Diagram not ready yet. Please wait for it to load completely.');
    }
  };

  window.exportDiagramAsPNG = function() {
    console.log('exportDiagramAsPNG called', window.dfdDiagram);
    if (window.dfdDiagram) {
      window.dfdDiagram.exportAsPNG();
    } else {
      alert('Diagram not ready yet. Please wait for it to load completely.');
    }
  };

  // Fallback simple export functions if D3 is not available
  window.exportFallbackSVG = function() {
    const container = document.getElementById('dfd-diagram-container');
    const svgElement = container ? container.querySelector('svg') : null;
    
    if (svgElement) {
      const svgData = new XMLSerializer().serializeToString(svgElement);
      const blob = new Blob([svgData], { type: 'image/svg+xml' });
      const url = URL.createObjectURL(blob);
      
      const link = document.createElement('a');
      link.download = 'dfd-diagram.svg';
      link.href = url;
      link.click();
      
      URL.revokeObjectURL(url);
    } else {
      alert('No diagram found to export.');
    }
  };

  window.exportFallbackPNG = function() {
    const container = document.getElementById('dfd-diagram-container');
    const svgElement = container ? container.querySelector('svg') : null;
    
    if (svgElement) {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const svgData = new XMLSerializer().serializeToString(svgElement);
      
      canvas.width = 1400;
      canvas.height = 800;
      
      const img = new Image();
      img.onload = function() {
        ctx.fillStyle = '#1a1a1a';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0);
        
        canvas.toBlob(blob => {
          const url = URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.download = 'dfd-diagram.png';
          link.href = url;
          link.click();
          URL.revokeObjectURL(url);
        }, 'image/png');
      };
      
      img.src = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
    } else {
      alert('No diagram found to export.');
    }
  };

  // Check if D3 is available
  if (typeof d3 === 'undefined') {
    console.error('D3.js is not loaded! The diagram will not work properly.');
    // Create a simple fallback message
    window.addEventListener('DOMContentLoaded', function() {
      const container = document.getElementById('dfd-diagram-container');
      if (container) {
        container.innerHTML = '<div style="padding: 20px; background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; text-align: center;"><h3>⚠️ Diagram Loading Issue</h3><p>The D3.js library failed to load. The diagram cannot be displayed.</p><p>Please check your internet connection and refresh the page.</p></div>';
      }
    });
  } else {
    console.log('D3.js loaded successfully, version:', d3.version);
  }

  // Drupal behavior
  Drupal.behaviors.d3DfdDiagram = {
    attach: function (context, settings) {
      console.log('D3 DFD Behavior attach called', {
        context: context,
        settings: settings,
        d3Available: typeof d3 !== 'undefined'
      });
      
      const diagramContainer = document.getElementById('dfd-diagram-container');
      
      if (diagramContainer && settings.dfdAnalyzer && settings.dfdAnalyzer.data) {
        console.log('Creating diagram with data:', settings.dfdAnalyzer.data);
        
        if (typeof d3 !== 'undefined') {
          try {
            // Create the diagram
            window.dfdDiagram = new DFDDiagram('#dfd-diagram-container', settings.dfdAnalyzer.data);
            console.log('Diagram created successfully');
            
            // Enable export buttons
            $('.export-png, .export-svg').prop('disabled', false).removeClass('button--disabled');
          } catch (error) {
            console.error('Error creating diagram:', error);
            diagramContainer.innerHTML = '<div style="padding: 20px; background: #f8cecc; border: 1px solid #b85450; border-radius: 4px; text-align: center;"><h3>❌ Diagram Error</h3><p>Failed to create diagram: ' + error.message + '</p></div>';
          }
        } else {
          console.error('D3.js not available, cannot create diagram');
          diagramContainer.innerHTML = '<div style="padding: 20px; background: #f8cecc; border: 1px solid #b85450; border-radius: 4px; text-align: center;"><h3>❌ D3.js Not Loaded</h3><p>The D3.js library is required but not available.</p></div>';
        }
      } else {
        console.warn('Missing requirements for diagram:', {
          container: !!diagramContainer,
          settings: !!settings.dfdAnalyzer,
          data: !!(settings.dfdAnalyzer && settings.dfdAnalyzer.data)
        });
      }
    }
  };

})(jQuery, Drupal);
