(function (Drupal) {
  'use strict';
  
  Drupal.behaviors.mermaidInit = {
    attach: function (context, settings) {
      // Check if already processed
      if (context !== document) {
        return;
      }
      
      console.log('Mermaid behavior attached');
      
      // Simple approach - wait for window load
      if (document.readyState === 'loading') {
        window.addEventListener('load', function() {
          setTimeout(initMermaid, 500);
        });
      } else {
        setTimeout(initMermaid, 500);
      }
      
      function initMermaid() {
        console.log('Attempting to initialize Mermaid...');
        console.log('Mermaid available:', typeof mermaid !== 'undefined');
        
        if (typeof mermaid === 'undefined') {
          console.error('Mermaid library not loaded!');
          // Show fallback message
          var diagrams = document.querySelectorAll('.mermaid');
          diagrams.forEach(function(diagram) {
            diagram.innerHTML = '<div style="color: red; padding: 20px; border: 1px solid red;">Mermaid library failed to load. Please check console for errors.</div>';
          });
          return;
        }
        
        try {
          // Initialize Mermaid
          mermaid.initialize({ 
            startOnLoad: false, 
            securityLevel: 'loose',
            theme: 'default',
            flowchart: { useMaxWidth: true }
          });
          
          console.log('Mermaid initialized, looking for diagrams...');
          
          // Find and render diagrams
          var diagrams = document.querySelectorAll('.mermaid');
          console.log('Found ' + diagrams.length + ' diagrams to process');
          
          diagrams.forEach(function(diagram, index) {
            if (diagram.hasAttribute('data-processed')) {
              return;
            }
            
            diagram.setAttribute('data-processed', 'true');
            var diagramText = diagram.textContent.trim();
            
            console.log('Processing diagram ' + index);
            console.log('Diagram text preview:', diagramText.substring(0, 200) + '...');
            
            // Use the render method
            mermaid.render('diagram-' + index, diagramText).then(function(result) {
              console.log('Diagram ' + index + ' rendered successfully');
              diagram.innerHTML = result.svg;
            }).catch(function(error) {
              console.error('Error rendering diagram ' + index + ':', error);
              diagram.innerHTML = '<div style="color: red; padding: 20px; border: 1px solid red;">Error: ' + error.message + '</div>';
            });
          });
          
        } catch (error) {
          console.error('Mermaid initialization error:', error);
          var diagrams = document.querySelectorAll('.mermaid');
          diagrams.forEach(function(diagram) {
            diagram.innerHTML = '<div style="color: red; padding: 20px; border: 1px solid red;">Error: ' + error.message + '</div>';
          });
        }
      }
    }
  };
})(Drupal);
