/**
 * @file
 * JavaScript for exporting Mermaid diagrams as images.
 */

(function ($, Drupal) {
  'use strict';

  // Helper function to get proper SVG dimensions
  function getSVGDimensions(svgElement) {
    let bbox, width, height;
    const padding = 20;
    
    try {
      // Try to get bounding box
      bbox = svgElement.getBBox();
      width = Math.max(bbox.width + (padding * 2), 800);
      height = Math.max(bbox.height + (padding * 2), 600);
      
      console.log('SVG getBBox dimensions:', bbox);
      
      // Special handling for very wide diagrams with compressed height
      if (width > height * 10) { // If width is more than 10x the height, it's likely compressed
        // Try to get all child elements and calculate proper bounds
        const allElements = svgElement.querySelectorAll('*');
        let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
        
        allElements.forEach(element => {
          try {
            const elementBBox = element.getBBox();
            if (elementBBox.width > 0 && elementBBox.height > 0) {
              minX = Math.min(minX, elementBBox.x);
              minY = Math.min(minY, elementBBox.y);
              maxX = Math.max(maxX, elementBBox.x + elementBBox.width);
              maxY = Math.max(maxY, elementBBox.y + elementBBox.height);
            }
          } catch (e) {
            // Skip elements that can't provide bbox
          }
        });
        
        if (isFinite(minX) && isFinite(minY) && isFinite(maxX) && isFinite(maxY)) {
          bbox = { x: minX, y: minY, width: maxX - minX, height: maxY - minY };
          width = Math.max(bbox.width + (padding * 2), 800);
          height = Math.max(bbox.height + (padding * 2), 600);
          console.log('Recalculated dimensions from elements:', bbox);
        }
      }
      
    } catch (e) {
      console.log('getBBox failed, using fallback:', e);
      // Fallback: use computed dimensions or viewport
      const rect = svgElement.getBoundingClientRect();
      const computedStyle = window.getComputedStyle(svgElement);
      
      width = rect.width || parseInt(computedStyle.width) || 
              parseInt(svgElement.getAttribute('width')) || 800;
      height = rect.height || parseInt(computedStyle.height) || 
               parseInt(svgElement.getAttribute('height')) || 600;
      
      // Add padding
      width = Math.max(width + (padding * 2), 800);
      height = Math.max(height + (padding * 2), 600);
      
      bbox = { x: 0, y: 0, width: width - padding * 2, height: height - padding * 2 };
    }
    
    console.log('Final dimensions:', { width, height, bbox });
    return { bbox, width, height, padding };
  }

  // Global functions for export functionality
  window.exportDiagramAsSVG = function() {
    const diagramElement = document.getElementById('mermaid-diagram');
    const svgElement = diagramElement.querySelector('svg');
    
    if (!svgElement) {
      alert('No diagram found to export.');
      return;
    }

    // Clone the SVG to avoid modifying the original
    const svgClone = svgElement.cloneNode(true);
    
    // Get proper dimensions
    const { bbox, width, height, padding } = getSVGDimensions(svgElement);
    
    // Set proper dimensions and viewBox
    svgClone.setAttribute('width', width);
    svgClone.setAttribute('height', height);
    svgClone.setAttribute('viewBox', `${bbox.x - padding} ${bbox.y - padding} ${width} ${height}`);
    
    // Ensure namespace is set
    svgClone.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
    svgClone.setAttribute('xmlns:xlink', 'http://www.w3.org/1999/xlink');
    
    // Add white background
    const backgroundRect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    backgroundRect.setAttribute('x', bbox.x - padding);
    backgroundRect.setAttribute('y', bbox.y - padding);
    backgroundRect.setAttribute('width', width);
    backgroundRect.setAttribute('height', height);
    backgroundRect.setAttribute('fill', 'white');
    svgClone.insertBefore(backgroundRect, svgClone.firstChild);
    
    // Get SVG content
    const svgData = new XMLSerializer().serializeToString(svgClone);
    const svgBlob = new Blob([svgData], {type: 'image/svg+xml;charset=utf-8'});
    
    // Create download link
    const downloadLink = document.createElement('a');
    downloadLink.href = URL.createObjectURL(svgBlob);
    downloadLink.download = 'dfd-diagram.svg';
    document.body.appendChild(downloadLink);
    downloadLink.click();
    document.body.removeChild(downloadLink);
  };

  window.exportDiagramAsPNG = function() {
    const diagramElement = document.getElementById('mermaid-diagram');
    const svgElement = diagramElement.querySelector('svg');
    
    if (!svgElement) {
      alert('No diagram found to export.');
      return;
    }

    // Clone the SVG and fix its dimensions (same as SVG export)
    const svgClone = svgElement.cloneNode(true);
    
    // Get proper dimensions using the helper function
    const { bbox, width: svgWidth, height: svgHeight, padding } = getSVGDimensions(svgElement);
    
    // Set proper dimensions and viewBox on the clone
    svgClone.setAttribute('width', svgWidth);
    svgClone.setAttribute('height', svgHeight);
    svgClone.setAttribute('viewBox', `${bbox.x - padding} ${bbox.y - padding} ${svgWidth} ${svgHeight}`);
    svgClone.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
    
    // Add white background to SVG
    const backgroundRect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    backgroundRect.setAttribute('x', bbox.x - padding);
    backgroundRect.setAttribute('y', bbox.y - padding);
    backgroundRect.setAttribute('width', svgWidth);
    backgroundRect.setAttribute('height', svgHeight);
    backgroundRect.setAttribute('fill', 'white');
    svgClone.insertBefore(backgroundRect, svgClone.firstChild);
    
    // Create a canvas element
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    
    // Set canvas size with scale factor for better quality
    const scaleFactor = 2;
    canvas.width = svgWidth * scaleFactor;
    canvas.height = svgHeight * scaleFactor;
    
    // Scale context to match
    ctx.scale(scaleFactor, scaleFactor);
    
    // Convert corrected SVG to data URL
    const svgData = new XMLSerializer().serializeToString(svgClone);
    const svgDataUrl = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
    
    // Create image from SVG
    const img = new Image();
    img.onload = function() {
      ctx.drawImage(img, 0, 0, svgWidth, svgHeight);
      
      // Convert canvas to PNG and download
      canvas.toBlob(function(blob) {
        const downloadLink = document.createElement('a');
        downloadLink.href = URL.createObjectURL(blob);
        downloadLink.download = 'dfd-diagram.png';
        document.body.appendChild(downloadLink);
        downloadLink.click();
        document.body.removeChild(downloadLink);
      }, 'image/png', 0.9);
    };
    
    img.onerror = function() {
      alert('Error converting SVG to PNG. Please try the SVG export instead.');
    };
    
    img.src = svgDataUrl;
  };

  // Function to set diagram size mode
  window.setDiagramSize = function(mode) {
    const diagramElement = document.getElementById('mermaid-diagram');
    const svgElement = diagramElement ? diagramElement.querySelector('svg') : null;
    
    if (!diagramElement || !svgElement) return;
    
    // Remove all size classes
    diagramElement.classList.remove('large-diagram', 'fit-diagram', 'scroll-diagram');
    
    // Update button states
    document.querySelectorAll('.size-control').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    
    switch(mode) {
      case 'fit':
        diagramElement.classList.add('fit-diagram');
        svgElement.style.maxWidth = '100%';
        svgElement.style.width = 'auto';
        svgElement.style.height = 'auto';
        break;
      case 'full':
        diagramElement.classList.add('large-diagram');
        svgElement.style.maxWidth = 'none';
        svgElement.style.width = 'auto';
        svgElement.style.height = 'auto';
        break;
      case 'scroll':
        diagramElement.classList.add('scroll-diagram');
        svgElement.style.maxWidth = 'none';
        svgElement.style.width = 'auto';
        svgElement.style.height = 'auto';
        break;
    }
  };

  // Function to check if diagram is large and suggest PNG export
  function checkDiagramSize() {
    const diagramElement = document.getElementById('mermaid-diagram');
    const svgElement = diagramElement ? diagramElement.querySelector('svg') : null;
    
    if (!svgElement) return;
    
    try {
      const { width, height } = getSVGDimensions(svgElement);
      
      // If diagram is large (>1500px width or >1000px height), show warning
      if (width > 1500 || height > 1000) {
        const warning = document.createElement('div');
        warning.className = 'large-diagram-warning';
        warning.style.cssText = 'background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; margin: 10px 0; border-radius: 4px; color: #856404;';
        warning.innerHTML = `
          <strong>Large Diagram Detected (${Math.round(width)}×${Math.round(height)}px)</strong><br>
          For better performance, consider using <strong>PNG export</strong> for sharing or documentation.
          <button onclick="this.parentElement.remove()" style="float: right; background: none; border: none; font-size: 16px; cursor: pointer;">×</button>
        `;
        
        const exportButtons = document.querySelector('.export-buttons');
        if (exportButtons && !document.querySelector('.large-diagram-warning')) {
          exportButtons.parentNode.insertBefore(warning, exportButtons);
        }
      }
    } catch (e) {
      console.log('Could not check diagram size:', e);
    }
  }

  // Function to optimize the SVG display on the page
  window.optimizeDiagramDisplay = function optimizeDiagramDisplay() {
    const diagramElement = document.getElementById('mermaid-diagram');
    const svgElement = diagramElement ? diagramElement.querySelector('svg') : null;
    
    if (!svgElement) return;
    
    try {
      console.log('Optimizing diagram display...');
      
      // Get current dimensions
      const currentViewBox = svgElement.getAttribute('viewBox');
      const currentWidth = svgElement.getAttribute('width');
      const currentHeight = svgElement.getAttribute('height');
      
      console.log('Current SVG attributes:', { currentViewBox, currentWidth, currentHeight });
      
      // Get proper dimensions
      const { bbox, width, height } = getSVGDimensions(svgElement);
      
      // Always fix compressed diagrams or missing viewBox
      const needsFix = !currentViewBox || 
                       currentHeight === '0' || 
                       parseInt(currentHeight) < 100 ||
                       (parseInt(currentWidth) > parseInt(currentHeight) * 10);
      
      if (needsFix) {
        console.log('Fixing compressed diagram...');
        
        // Remove any problematic attributes
        svgElement.removeAttribute('style');
        
        // Set proper dimensions and viewBox
        svgElement.setAttribute('viewBox', `${bbox.x - 10} ${bbox.y - 10} ${width} ${height}`);
        svgElement.setAttribute('width', width);
        svgElement.setAttribute('height', height);
        
        // Force proper CSS
        svgElement.style.cssText = '';
        svgElement.style.width = '100%';
        svgElement.style.height = 'auto';
        svgElement.style.maxWidth = '100%';
        svgElement.style.display = 'block';
        
        console.log('Applied fix with dimensions:', { width, height, bbox });
        
        // Also fix the container
        diagramElement.style.overflow = 'auto';
        diagramElement.style.maxHeight = '70vh';
        
        // Trigger a reflow
        svgElement.style.display = 'none';
        svgElement.offsetHeight; // Force reflow
        svgElement.style.display = 'block';
      }
      
      // Check if diagram is large and show warning
      checkDiagramSize();
    } catch (e) {
      console.log('Could not optimize diagram display:', e);
    }
  };

  // Initialize when DOM is ready
  Drupal.behaviors.diagramExport = {
    attach: function (context, settings) {
      // Add some styling for export buttons
      $('.export-buttons', context).once('styled').each(function() {
        $(this).css({
          'margin': '10px 0',
          'padding': '10px',
          'background': '#f8f9fa',
          'border-radius': '4px'
        });
        
        $(this).find('.button').css({
          'margin-right': '10px',
          'margin-bottom': '5px'
        });
      });
      
      // Ensure Mermaid diagram is properly rendered before allowing exports
      if (typeof mermaid !== 'undefined') {
        // Wait for mermaid to finish rendering
        const checkMermaidReady = function() {
          const diagramElement = document.getElementById('mermaid-diagram');
          if (diagramElement && diagramElement.querySelector('svg')) {
            // Optimize the diagram display
            optimizeDiagramDisplay();
            
            // Enable export buttons
            $('.export-png, .export-svg').prop('disabled', false).removeClass('button--disabled');
          } else {
            // Check again in 100ms
            setTimeout(checkMermaidReady, 100);
          }
        };
        
        // Initially disable buttons until diagram is ready
        $('.export-png, .export-svg').prop('disabled', true).addClass('button--disabled');
        setTimeout(checkMermaidReady, 500);
      }
    }
  };

})(jQuery, Drupal);