/**
 * @file
 * Standalone export functions that work immediately.
 */

(function() {
  'use strict';

  console.log('Loading standalone export functions...');

  // Define export functions immediately
  window.exportDiagramAsSVG = function() {
    console.log('Export SVG function called');
    
    // Check if D3 diagram exists
    if (window.dfdDiagram && typeof window.dfdDiagram.exportAsSVG === 'function') {
      console.log('Using D3 diagram export');
      window.dfdDiagram.exportAsSVG();
      return;
    }
    
    // Fallback export
    console.log('Using fallback SVG export');
    const container = document.getElementById('dfd-diagram-container');
    const svgElement = container ? container.querySelector('svg') : null;
    
    if (svgElement) {
      const svgData = new XMLSerializer().serializeToString(svgElement);
      const blob = new Blob([svgData], { type: 'image/svg+xml' });
      const url = URL.createObjectURL(blob);
      
      const link = document.createElement('a');
      link.download = 'dfd-diagram.svg';
      link.href = url;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      
      URL.revokeObjectURL(url);
    } else {
      alert('No diagram found to export. Please wait for the diagram to load.');
    }
  };

  window.exportDiagramAsPNG = function() {
    console.log('Export PNG function called');
    
    // Check if D3 diagram exists
    if (window.dfdDiagram && typeof window.dfdDiagram.exportAsPNG === 'function') {
      console.log('Using D3 diagram export');
      window.dfdDiagram.exportAsPNG();
      return;
    }
    
    // Fallback export
    console.log('Using fallback PNG export');
    const container = document.getElementById('dfd-diagram-container');
    const svgElement = container ? container.querySelector('svg') : null;
    
    if (svgElement) {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const svgData = new XMLSerializer().serializeToString(svgElement);
      
      canvas.width = 1400;
      canvas.height = 800;
      
      const img = new Image();
      img.onload = function() {
        ctx.fillStyle = '#1a1a1a';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0);
        
        canvas.toBlob(blob => {
          const url = URL.createObjectURL(blob);
          const link = document.createElement('a');
          link.download = 'dfd-diagram.png';
          link.href = url;
          document.body.appendChild(link);
          link.click();
          document.body.removeChild(link);
          URL.revokeObjectURL(url);
        }, 'image/png');
      };
      
      img.onerror = function() {
        alert('Error converting diagram to PNG. Please try SVG export instead.');
      };
      
      img.src = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
    } else {
      alert('No diagram found to export. Please wait for the diagram to load.');
    }
  };

  // Notify that functions are ready
  console.log('Standalone export functions loaded successfully');
  
  // Add a flag so we know these are loaded
  window.exportFunctionsReady = true;

})();