/**
 * @file
 * Simple HTML/CSS based DFD diagram (no D3.js dependency).
 */

(function ($, Drupal) {
  'use strict';

  console.log('Loading simple diagram generator...');

  // Drupal behavior for simple diagram
  Drupal.behaviors.simpleDfdDiagram = {
    attach: function (context, settings) {
      console.log('Simple DFD behavior attach called', settings);
      
      const diagramContainer = document.getElementById('dfd-diagram-container');
      
      if (diagramContainer && settings.dfdAnalyzer && settings.dfdAnalyzer.data) {
        console.log('Creating simple diagram with data:', settings.dfdAnalyzer.data);
        
        try {
          createSimpleDiagram(
            diagramContainer,
            settings.dfdAnalyzer.data,
            settings.dfdAnalyzer.presentation
          );
          
          // Enable export buttons
          $('.export-png, .export-svg').prop('disabled', false).removeClass('button--disabled');
        } catch (error) {
          console.error('Error creating simple diagram:', error);
          diagramContainer.innerHTML = '<div class="diagram-error">Failed to create diagram: ' + error.message + '</div>';
        }
      }
    }
  };

  function createSimpleDiagram(container, data, presentation) {
    console.log('Creating simple diagram with column layout...');
    
    // Clear container
    container.innerHTML = '';
    window.diagramNodePositions = {};
    container.classList.add('dfd-column-layout');
    
    const nodeWidth = 220;
    const nodeHeight = 150;
    const columnGap = 80;
    const rowGap = 30;
    const startX = 40;
    const startY = 40;
    
    const entityGroups = buildEntityGroups(presentation, data);
    
    console.log('Entity groups for column layout:', entityGroups.map(group => ({
      type: group.entityType,
      count: group.entries.length
    })));
    
    if (!entityGroups.length) {
      container.innerHTML = '<div class="diagram-error">No entity data available to render.</div>';
      return;
    }
    
    const columnCount = entityGroups.length;
    const maxRows = Math.max(...entityGroups.map(group => group.entries.length));
    const svgWidth = Math.max(
      600,
      startX * 2 + columnCount * nodeWidth + Math.max(0, columnCount - 1) * columnGap
    );
    const svgHeight = Math.max(
      600,
      startY * 2 + maxRows * nodeHeight + Math.max(0, maxRows - 1) * rowGap
    );
    
    // Create SVG
    const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svg.setAttribute('width', svgWidth);
    svg.setAttribute('height', svgHeight);
    svg.setAttribute('viewBox', `0 0 ${svgWidth} ${svgHeight}`);
    svg.style.background = '#1a1a1a';
    svg.style.border = '1px solid #333';
    svg.style.borderRadius = '6px';
    
    entityGroups.forEach((group, columnIndex) => {
      const columnX = startX + columnIndex * (nodeWidth + columnGap);
      
      group.entries.forEach((entry, rowIndex) => {
        const rowY = startY + rowIndex * (nodeHeight + rowGap);
        const fields = group.includeFields === false ? [] : (entry.fields || []);
        createNode(svg, {
          id: entry.id,
          type: group.type,
          entityType: group.entityType,
          bundle: entry.bundle,
          fields,
          x: columnX,
          y: rowY,
          width: nodeWidth,
          height: nodeHeight
        });
      });
    });
    
    container.appendChild(svg);
    console.log('Simple diagram created successfully');
  }
  
  function createNode(svg, nodeData) {
    // Store position for relationship drawing
    window.diagramNodePositions[nodeData.id] = { x: nodeData.x, y: nodeData.y };
    
    const group = document.createElementNS('http://www.w3.org/2000/svg', 'g');
    group.setAttribute('transform', `translate(${nodeData.x}, ${nodeData.y})`);
    
    const width = nodeData.width || 220;
    const height = nodeData.height || 150;
    
    // Node background
    const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    rect.setAttribute('width', width);
    rect.setAttribute('height', height);
    rect.setAttribute('rx', '8');
    rect.setAttribute('fill', '#2d3436');
    rect.setAttribute('stroke', getNodeBorderColor(nodeData.type));
    rect.setAttribute('stroke-width', '2');
    group.appendChild(rect);
    
    // Header background
    const headerRect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    headerRect.setAttribute('width', width);
    headerRect.setAttribute('height', '30');
    headerRect.setAttribute('rx', '8');
    headerRect.setAttribute('fill', getNodeBorderColor(nodeData.type));
    group.appendChild(headerRect);
    
    // Header fix for bottom corners
    const headerFix = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
    headerFix.setAttribute('y', '25');
    headerFix.setAttribute('width', width);
    headerFix.setAttribute('height', '8');
    headerFix.setAttribute('fill', getNodeBorderColor(nodeData.type));
    group.appendChild(headerFix);
    
    // Entity type label
    const entityLabel = document.createElementNS('http://www.w3.org/2000/svg', 'text');
    entityLabel.setAttribute('x', '12');
    entityLabel.setAttribute('y', '22');
    entityLabel.setAttribute('fill', '#ffffff');
    entityLabel.setAttribute('font-size', '13');
    entityLabel.setAttribute('font-weight', 'bold');
    entityLabel.setAttribute('font-family', 'monospace');
    entityLabel.textContent = nodeData.entityType;
    group.appendChild(entityLabel);
    
    // Bundle name
    const bundleLabel = document.createElementNS('http://www.w3.org/2000/svg', 'text');
    bundleLabel.setAttribute('x', '65');
    bundleLabel.setAttribute('y', '22');
    bundleLabel.setAttribute('fill', '#ffffff');
    bundleLabel.setAttribute('font-size', '13');
    bundleLabel.setAttribute('font-family', 'monospace');
    const cleanBundleName = nodeData.bundle.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    bundleLabel.textContent = cleanBundleName.length > 15 ? cleanBundleName.substring(0, 15) + '...' : cleanBundleName;
    group.appendChild(bundleLabel);
    
    const displayFields = normalizeFieldList(nodeData.fields);
    if (displayFields.length) {
      let yOffset = 50;
      const maxFieldsToShow = 4;
      displayFields.slice(0, maxFieldsToShow).forEach((fieldName) => {
        const bullet = document.createElementNS('http://www.w3.org/2000/svg', 'text');
        bullet.setAttribute('x', '12');
        bullet.setAttribute('y', yOffset);
        bullet.setAttribute('fill', 'rgba(255, 255, 255, 0.9)');
        bullet.setAttribute('font-size', '10');
        bullet.setAttribute('font-family', 'Arial, sans-serif');
        bullet.textContent = '•';
        group.appendChild(bullet);
        
        const fieldLabel = document.createElementNS('http://www.w3.org/2000/svg', 'text');
        fieldLabel.setAttribute('x', '25');
        fieldLabel.setAttribute('y', yOffset);
        fieldLabel.setAttribute('fill', 'rgba(255, 255, 255, 0.9)');
        fieldLabel.setAttribute('font-size', '10');
        fieldLabel.setAttribute('font-family', 'Arial, sans-serif');
        const truncatedName = fieldName.length > 20 ? fieldName.substring(0, 20) + '...' : fieldName;
        fieldLabel.textContent = truncatedName;
        group.appendChild(fieldLabel);
        
        yOffset += 12;
      });
      
      if (displayFields.length > maxFieldsToShow) {
        const moreBullet = document.createElementNS('http://www.w3.org/2000/svg', 'text');
        moreBullet.setAttribute('x', '12');
        moreBullet.setAttribute('y', yOffset);
        moreBullet.setAttribute('fill', 'rgba(255, 255, 255, 0.7)');
        moreBullet.setAttribute('font-size', '10');
        moreBullet.setAttribute('font-family', 'Arial, sans-serif');
        moreBullet.textContent = '•';
        group.appendChild(moreBullet);
        
        const moreLabel = document.createElementNS('http://www.w3.org/2000/svg', 'text');
        moreLabel.setAttribute('x', '25');
        moreLabel.setAttribute('y', yOffset);
        moreLabel.setAttribute('fill', 'rgba(255, 255, 255, 0.7)');
        moreLabel.setAttribute('font-size', '10');
        moreLabel.setAttribute('font-family', 'Arial, sans-serif');
        moreLabel.textContent = '...';
        group.appendChild(moreLabel);
      }
    }
    else {
      const placeholder = document.createElementNS('http://www.w3.org/2000/svg', 'text');
      placeholder.setAttribute('x', '12');
      placeholder.setAttribute('y', '60');
      placeholder.setAttribute('fill', 'rgba(255, 255, 255, 0.6)');
      placeholder.setAttribute('font-size', '10');
      placeholder.setAttribute('font-family', 'Arial, sans-serif');
      placeholder.textContent = 'No highlighted fields';
      group.appendChild(placeholder);
    }
    
    svg.appendChild(group);
  }
  
  function buildEntityGroups(presentation, data) {
    if (presentation && Array.isArray(presentation.groups) && presentation.groups.length) {
      return presentation.groups.map(group => {
        const includeFields = group.includeFields !== false;
        return {
          type: group.node_type || group.collection || 'content_type',
          entityType: group.entity_type_label || 'Entity',
          includeFields,
          entries: (group.entities || []).map(entity => ({
            id: entity.id,
            bundle: entity.label || entity.bundle || entity.id,
            fields: resolveEntityFields(entity, data)
          }))
        };
      }).filter(group => group.entries.length);
    }
    
    return buildGroupsFromRaw(data);
  }
  
  function buildGroupsFromRaw(data = {}) {
    const sortEntries = (items = {}) => {
      return Object.entries(items)
        .sort(([, a], [, b]) => {
          const labelA = (a.label || a.bundle || '').toLowerCase();
          const labelB = (b.label || b.bundle || '').toLowerCase();
          return labelA.localeCompare(labelB);
        })
        .map(([bundleId, bundle]) => ({
          id: bundleId,
          bundle: bundle.bundle || bundle.label || bundleId,
          fields: bundle.fields || {}
        }));
      };
    
    return [
      {
        type: 'content_type',
        entityType: 'CT',
        includeFields: true,
        entries: sortEntries(data.bundles || {})
      },
      {
        type: 'paragraph',
        entityType: 'Para',
        includeFields: true,
        entries: sortEntries(data.paragraphs || {})
      },
      {
        type: 'taxonomy',
        entityType: 'Voc',
        includeFields: true,
        entries: sortEntries(data.taxonomies || {}).map(entry => ({ ...entry, fields: null }))
      },
      {
        type: 'media',
        entityType: 'Media',
        includeFields: true,
        entries: sortEntries(data.media || {})
      },
      {
        type: 'webform',
        entityType: 'Form',
        includeFields: true,
        entries: sortEntries(data.webforms || {})
      }
    ].filter(group => group.entries.length);
  }

  function resolveEntityFields(entity, data) {
    const existing = entity.fields;
    if (Array.isArray(existing) && existing.length) {
      return existing;
    }
    if (existing && typeof existing === 'object' && Object.keys(existing).length) {
      return existing;
    }

    return getRawEntityFields(data, entity.id);
  }

  function getRawEntityFields(data = {}, entityId) {
    if (!entityId) {
      return {};
    }
    const pools = ['bundles', 'paragraphs', 'taxonomies', 'media', 'webforms'];
    for (const pool of pools) {
      if (data[pool] && data[pool][entityId] && data[pool][entityId].fields) {
        return data[pool][entityId].fields;
      }
    }
    return {};
  }
  
  function normalizeFieldList(fields) {
    if (!fields) {
      return [];
    }
    
    if (Array.isArray(fields)) {
      return fields;
    }
    
    return Object.keys(fields)
      .filter((fieldName) => fieldName.startsWith('field_') || ['body', 'title'].includes(fieldName))
      .map((fieldName) => fieldName
        .replace(/^field_/, '')
        .replace(/_/g, ' ')
        .replace(/\b\w/g, (char) => char.toUpperCase())
      );
  }
  
  function getNodeBorderColor(type) {
    const colors = {
      'content_type': '#4a90e2',
      'paragraph': '#7b68ee',
      'taxonomy': '#ff6b6b',
      'media': '#26d0ce',
      'webform': '#ffa500'
    };
    return colors[type] || '#666';
  }
  
  // Store node positions for relationship drawing
  window.diagramNodePositions = {};
  
  function addRelationships(svg, data) {
    // Add arrow marker definition
    const defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
    const marker = document.createElementNS('http://www.w3.org/2000/svg', 'marker');
    marker.setAttribute('id', 'arrowhead');
    marker.setAttribute('viewBox', '0 -5 10 10');
    marker.setAttribute('refX', '8');
    marker.setAttribute('refY', '0');
    marker.setAttribute('markerWidth', '8');
    marker.setAttribute('markerHeight', '8');
    marker.setAttribute('orient', 'auto');
    
    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    path.setAttribute('d', 'M0,-5L10,0L0,5');
    path.setAttribute('fill', '#ff6b9d');
    marker.appendChild(path);
    defs.appendChild(marker);
    svg.insertBefore(defs, svg.firstChild);
    
    // Draw relationship lines
    (data.relationships || []).forEach(rel => {
      const sourcePos = window.diagramNodePositions[rel.source];
      if (!sourcePos) return;
      
      (rel.target_bundles || []).forEach(targetBundle => {
        const targetKey = rel.target_entity_type + '.' + targetBundle;
        const targetPos = window.diagramNodePositions[targetKey];
        
        if (targetPos) {
          // Create line
          const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
          line.setAttribute('x1', sourcePos.x + 100); // Center of source node
          line.setAttribute('y1', sourcePos.y + 60);
          line.setAttribute('x2', targetPos.x + 100); // Center of target node
          line.setAttribute('y2', targetPos.y + 60);
          line.setAttribute('stroke', '#ff6b9d');
          line.setAttribute('stroke-width', '2');
          line.setAttribute('marker-end', 'url(#arrowhead)');
          line.setAttribute('opacity', '0.7');
          svg.appendChild(line);
          
          // Add field label
          const midX = (sourcePos.x + targetPos.x + 200) / 2;
          const midY = (sourcePos.y + targetPos.y + 120) / 2;
          
          const label = document.createElementNS('http://www.w3.org/2000/svg', 'text');
          label.setAttribute('x', midX);
          label.setAttribute('y', midY - 5);
          label.setAttribute('text-anchor', 'middle');
          label.setAttribute('fill', '#ffffff');
          label.setAttribute('font-size', '10');
          label.setAttribute('font-family', 'monospace');
          label.textContent = rel.field.replace('field_', '');
          svg.appendChild(label);
        }
      });
    });
  }

})(jQuery, Drupal);
