<?php

namespace Drupal\drupal_site_dfd_analyzer\Controller;

use Drupal\Component\Utility\Html;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\drupal_site_dfd_analyzer\Service\StructureAnalyzer;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;

class DfdReportController extends ControllerBase {

  /** @var \Drupal\drupal_site_dfd_analyzer\Service\StructureAnalyzer */
  protected $analyzer;

  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->analyzer = $container->get('drupal_site_dfd_analyzer.structure_analyzer');
    return $instance;
  }

  public function report() {
    $data = $this->analyzer->analyze();
    $presentation = $this->buildPresentationGroups($data);
    $csvContent = $this->generateCsv($data, $presentation);

    $build = [
      '#type' => 'container',
      'title' => [
        '#markup' => '<h2>Project DFD</h2>',
      ],
      'desc' => [
        '#markup' => '<p>This diagram shows entity relationships based on entity reference fields.</p>',
      ],
      'export_buttons' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['export-buttons', 'clearfix']],
        'download_png' => [
          '#type' => 'button',
          '#value' => $this->t('📷 Download as PNG'),
          '#attributes' => [
            'class' => ['button', 'button--primary', 'export-png'],
            'onclick' => 'exportDiagramAsPNG()',
          ],
        ],
        'download_svg' => [
          '#type' => 'button',
          '#value' => $this->t('📄 Download as SVG'),
          '#attributes' => [
            'class' => ['button', 'export-svg'],
            'onclick' => 'exportDiagramAsSVG()',
          ],
        ],
        'drawio_link' => [
          '#type' => 'link',
          '#title' => $this->t('🎨 Open in Draw.io'),
          '#url' => \Drupal\Core\Url::fromRoute('drupal_site_dfd_analyzer.drawio_export'),
          '#attributes' => [
            'class' => ['button', 'button--secondary'],
            'target' => '_blank',
          ],
        ],
        'download_csv' => [
          '#type' => 'link',
          '#title' => $this->t('📑 Download CSV'),
          '#url' => \Drupal\Core\Url::fromRoute('drupal_site_dfd_analyzer.csv_export'),
          '#attributes' => [
            'class' => ['button', 'button--secondary'],
          ],
        ],
      ],
      'diagram' => [
        '#markup' => '<div id="dfd-diagram-container" class="dfd-diagram-container"></div>',
      ],
      'raw' => [
        '#type' => 'details',
        '#title' => $this->t('Raw JSON Data'),
        '#open' => FALSE,
        'pre' => [
          '#markup' => '<pre>' . htmlspecialchars(json_encode([
            'data' => $data,
            'presentation' => $presentation,
          ], JSON_PRETTY_PRINT), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</pre>',
        ],
      ],
      'csv_raw' => [
        '#type' => 'details',
        '#title' => $this->t('Relationship CSV Data'),
        '#open' => FALSE,
        'pre' => [
          '#markup' => '<pre>' . htmlspecialchars($csvContent, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</pre>',
        ],
      ],
      '#attached' => [
        'library' => [
          'drupal_site_dfd_analyzer/simple_diagram',
        ],
        'drupalSettings' => [
          'dfdAnalyzer' => [
            'data' => $data,
            'presentation' => $presentation,
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Generate and return draw.io compatible XML.
   */
  public function drawioExport() {
    $data = $this->analyzer->analyze();
    $presentation = $this->buildPresentationGroups($data);
    $xml = $this->generateDrawioXML($data, $presentation);
    
    $response = new Response($xml);
    $response->headers->set('Content-Type', 'application/xml');
    $response->headers->set('Content-Disposition', 'attachment; filename="dfd_diagram.xml"');
    
    return $response;
  }

  /**
   * Export the relationships data as CSV.
   */
  public function csvExport() {
    $data = $this->analyzer->analyze();
    $presentation = $this->buildPresentationGroups($data);
    $csv = $this->generateCsv($data, $presentation);

    $response = new Response($csv);
    $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="dfd_relationships.csv"');

    return $response;
  }

  /**
   * Get draw.io style for different entity types.
   */
  private function getDrawioStyle($entityType) {
    $visual = $this->getEntityVisualDefinition($entityType);
    $borderColor = $visual['border'];
    
    return 'rounded=1;whiteSpace=wrap;html=1;align=left;verticalAlign=top;spacing=0;strokeWidth=2;strokeColor=' . $borderColor . ';fillColor=#2d3436;fontColor=#ffffff;fontSize=12;';
  }

  /**
   * Return visual config for each entity type.
   */
  private function getEntityVisualDefinition(string $entityType): array {
    $map = [
      'bundles' => ['short' => 'CT', 'header' => '#4A90E2', 'border' => '#4A90E2'],
      'paragraphs' => ['short' => 'Para', 'header' => '#7B68EE', 'border' => '#7B68EE'],
      'taxonomies' => ['short' => 'Voc', 'header' => '#FF6B6B', 'border' => '#FF6B6B'],
      'media' => ['short' => 'Media', 'header' => '#26D0CE', 'border' => '#26D0CE'],
      'webforms' => ['short' => 'Form', 'header' => '#FFA500', 'border' => '#FFA500'],
    ];
    
    if (isset($map[$entityType])) {
      return $map[$entityType];
    }
    
    $fallbackLabel = strtoupper(substr($entityType, 0, 3));
    return ['short' => $fallbackLabel, 'header' => '#555555', 'border' => '#555555'];
  }

  /**
   * Build the HTML label used for draw.io nodes.
   */
  private function buildDrawioLabel(string $entityShortLabel, string $bundleName, array $fieldNames, string $headerColor): string {
    $header = '<div style="background:' . Html::escape($headerColor) . ';color:#ffffff;padding:6px 10px;font-size:12px;font-weight:600;border-top-left-radius:6px;border-top-right-radius:6px;letter-spacing:0.5px;">'
      . '<span style="font-weight:700;margin-right:8px;">' . Html::escape($entityShortLabel) . '</span>'
      . Html::escape($bundleName)
      . '</div>';
    
    $body = '';
    $displayFields = array_slice($fieldNames, 0, 6);
    foreach ($displayFields as $field) {
      $body .= '<div style="color:#e6e6e6;font-size:11px;line-height:1.4;margin-bottom:2px;">&#8226; ' . Html::escape($field) . '</div>';
    }
    
    if (count($fieldNames) > count($displayFields)) {
      $body .= '<div style="color:#999;font-size:11px;line-height:1.4;">&#8226; ...</div>';
    }
    
    if ($body === '') {
      $body = '<div style="color:#999;font-size:11px;">No highlighted fields</div>';
    }
    
    $html = '<div style="font-family:\'Roboto\', Arial, sans-serif;">' . $header .
      '<div style="padding:10px 12px;">' . $body . '</div></div>';
    
    return htmlspecialchars($html, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  }

  /**
   * Extract relevant field labels for a node.
   */
  private function extractFieldNames(array $fields): array {
    $names = [];
    
    foreach ($fields as $fieldName => $definition) {
      if (!is_string($fieldName)) {
        continue;
      }
      if (strpos($fieldName, 'field_') === 0 || in_array($fieldName, ['body', 'title'], TRUE)) {
        $names[] = $this->formatFieldLabel($fieldName);
      }
    }
    
    return $names;
  }

  /**
   * Clean up field labels for display.
   */
  private function formatFieldLabel(string $fieldName): string {
    $clean = preg_replace('/^field_/', '', $fieldName);
    $clean = str_replace('_', ' ', $clean);
    return ucwords(trim($clean));
  }

  /**
   * Sort entity arrays alphabetically by label/bundle.
   */
  private function sortEntityList(array $entities): array {
    uasort($entities, function ($a, $b) {
      $labelA = strtolower($a['label'] ?? $a['bundle'] ?? '');
      $labelB = strtolower($b['label'] ?? $b['bundle'] ?? '');
      return $labelA <=> $labelB;
    });
    return $entities;
  }

  /**
   * Build grouped presentation metadata shared across UI/exports.
   */
  private function buildPresentationGroups(array $data): array {
    $definitions = [
      'bundles' => ['node_type' => 'content_type'],
      'paragraphs' => ['node_type' => 'paragraph'],
      'taxonomies' => ['node_type' => 'taxonomy'],
      'media' => ['node_type' => 'media'],
      'webforms' => ['node_type' => 'webform'],
    ];
    
    $groups = [];
    
    foreach ($definitions as $collection => $definition) {
      $entities = $this->sortEntityList($data[$collection] ?? []);
      if (empty($entities)) {
        continue;
      }
      
      $visual = $this->getEntityVisualDefinition($collection);
      $group = [
        'collection' => $collection,
        'node_type' => $definition['node_type'],
        'entity_type_label' => $visual['short'],
        'header_color' => $visual['header'],
        'border_color' => $visual['border'],
        'entities' => [],
      ];
      
      foreach ($entities as $entityId => $entity) {
        $bundleName = $entity['bundle'] ?? $entity['label'] ?? $entityId;
        $group['entities'][] = [
          'id' => $entityId,
          'bundle' => $bundleName,
          'label' => str_replace('_', ' ', ucwords($bundleName, '_')),
          'fields' => $this->extractFieldNames($entity['fields'] ?? []),
        ];
      }
      
      $groups[] = $group;
    }
    
    return [
      'generated' => date('c'),
      'groups' => $groups,
    ];
  }

  /**
   * Generate draw.io XML format from analyzed data.
   */
  private function generateDrawioXML(array $data, array $presentation): string {
    // Generate a simpler, working XML structure
    $xmlContent = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xmlContent .= '<mxfile host="app.diagrams.net" modified="' . date('c') . '" agent="Drupal DFD Analyzer" version="24.7.17">' . "\n";
    $xmlContent .= '  <diagram id="dfd_' . uniqid() . '" name="Project DFD">' . "\n";
    
    // Build the internal graph model as plain XML
    $graphContent = '<mxGraphModel dx="1426" dy="790" grid="1" gridSize="10" guides="1" tooltips="1" connect="1" arrows="1" fold="1" page="1" pageScale="1" pageWidth="827" pageHeight="1169" math="0" shadow="0">';
    $graphContent .= '<root>';
    $graphContent .= '<mxCell id="0"/>';
    $graphContent .= '<mxCell id="1" parent="0"/>';
    
    $cellId = 2;
    $positions = [];
    
    $nodeWidth = 220;
    $nodeHeight = 150;
    $columnGap = 100;
    $rowGap = 40;
    $startX = 30;
    $startY = 30;
    
    $groups = $presentation['groups'] ?? [];
    
    foreach ($groups as $columnIndex => $group) {
      $rowIndex = 0;
      foreach ($group['entities'] as $entity) {
        $x = $startX + $columnIndex * ($nodeWidth + $columnGap);
        $y = $startY + $rowIndex * ($nodeHeight + $rowGap);
        $rowIndex++;
        
        $nodeId = $group['collection'] . '_' . $cellId++;
        $positions[$entity['id']] = $nodeId;
        
        $label = $this->buildDrawioLabel(
          $group['entity_type_label'],
          $entity['label'],
          $entity['fields'],
          $group['header_color']
        );
        $style = $this->getDrawioStyle($group['collection']);
        
        $graphContent .= '<mxCell id="' . $nodeId . '" value="' . $label . '" style="' . $style . '" vertex="1" parent="1">';
        $graphContent .= '<mxGeometry x="' . $x . '" y="' . $y . '" width="' . $nodeWidth . '" height="' . $nodeHeight . '" as="geometry"/>';
        $graphContent .= '</mxCell>';
      }
    }
    
    // Add only the most important relationship edges to avoid clutter
    $addedRelationships = [];
    foreach ($data['relationships'] as $rel) {
      $sourceId = $positions[$rel['source']] ?? null;
      
      if ($sourceId && !empty($rel['target_bundles'])) {
        foreach ($rel['target_bundles'] as $targetBundle) {
          $targetKey = $rel['target_entity_type'] . '.' . $targetBundle;
          $targetId = $positions[$targetKey] ?? null;
          
          // Avoid duplicate relationships
          $relationshipKey = $sourceId . '->' . $targetId;
          if ($targetId && !isset($addedRelationships[$relationshipKey])) {
            $edgeId = 'edge_' . $cellId++;
            $graphContent .= '<mxCell id="' . $edgeId . '" value="" style="endArrow=classic;html=1;rounded=0;fontSize=10;strokeColor=#FF6B9D;" edge="1" parent="1" source="' . $sourceId . '" target="' . $targetId . '">';
            $graphContent .= '<mxGeometry relative="1" as="geometry"/>';
            $graphContent .= '</mxCell>';
            
            $addedRelationships[$relationshipKey] = true;
          }
        }
      }
    }
    
    $graphContent .= '</root>';
    $graphContent .= '</mxGraphModel>';
    
    // Encode the content for draw.io
    $encodedContent = base64_encode(gzdeflate($graphContent, 9));
    
    $xmlContent .= '    ' . $encodedContent . "\n";
    $xmlContent .= '  </diagram>' . "\n";
    $xmlContent .= '</mxfile>';

    return $xmlContent;
  }

  /**
   * Generate a CSV string representing entity relationships.
   */
  private function generateCsv(array $data, array $presentation = []): string {
    $handle = fopen('php://temp', 'w+');
    if ($handle === FALSE) {
      return '';
    }

    $groups = $presentation['groups'] ?? [];
    if (!empty($groups)) {
      fputcsv($handle, ['Entity Type', 'Bundle', 'Label', 'Highlighted Fields']);
      foreach ($groups as $group) {
        foreach ($group['entities'] as $entity) {
          fputcsv($handle, [
            $group['entity_type_label'],
            $entity['bundle'],
            $entity['label'],
            implode(', ', $entity['fields']),
          ]);
        }
      }
      fputcsv($handle, []); // spacer
    }

    fputcsv($handle, [
      'Source Entity Type',
      'Source Bundle',
      'Source Label',
      'Field',
      'Field Type',
      'Target Entity Type',
      'Target Bundle',
      'Target Label',
    ]);

    foreach ($data['relationships'] as $relationship) {
      $sourceInfo = $this->getEntityInfo($data, $relationship['source']);
      $targetBundles = $relationship['target_bundles'] ?? [];

      if (!empty($targetBundles)) {
        foreach ($targetBundles as $targetBundle) {
          $targetKey = $relationship['target_entity_type'] . '.' . $targetBundle;
          $targetInfo = $this->getEntityInfo($data, $targetKey, $relationship['target_entity_type']);
          fputcsv($handle, [
            $sourceInfo['entity_type'],
            $sourceInfo['bundle'],
            $sourceInfo['label'],
            $relationship['field'],
            $relationship['type'],
            $targetInfo['entity_type'],
            $targetInfo['bundle'],
            $targetInfo['label'],
          ]);
        }
      }
      else {
        $targetInfo = $this->getEntityInfo($data, $relationship['target_entity_type'], $relationship['target_entity_type']);
        fputcsv($handle, [
          $sourceInfo['entity_type'],
          $sourceInfo['bundle'],
          $sourceInfo['label'],
          $relationship['field'],
          $relationship['type'],
          $targetInfo['entity_type'],
          '',
          $targetInfo['label'],
        ]);
      }
    }

    rewind($handle);
    $csv = stream_get_contents($handle);
    fclose($handle);

    return $csv ?: '';
  }

  /**
   * Resolve human-readable info for an entity identifier.
   */
  private function getEntityInfo(array $data, string $key, ?string $fallbackEntityType = NULL): array {
    foreach (['bundles', 'paragraphs', 'taxonomies', 'media', 'webforms'] as $collection) {
      if (isset($data[$collection][$key])) {
        $entity = $data[$collection][$key];
        return [
          'entity_type' => $entity['entity_type'] ?? $fallbackEntityType ?? $collection,
          'bundle' => $entity['bundle'] ?? '',
          'label' => $entity['label'] ?? ($entity['bundle'] ?? $key),
        ];
      }
    }

    if (strpos($key, '.') !== FALSE) {
      [$entityType, $bundle] = explode('.', $key, 2);
      return [
        'entity_type' => $entityType,
        'bundle' => $bundle,
        'label' => ucwords(str_replace('_', ' ', $bundle)),
      ];
    }

    return [
      'entity_type' => $fallbackEntityType ?? $key,
      'bundle' => '',
      'label' => ucwords(str_replace('_', ' ', $key)),
    ];
  }
}
