<?php

namespace Drupal\drupal_site_dfd_analyzer\Drush\Commands;

use Drush\Commands\DrushCommands;
use Drupal\drupal_site_dfd_analyzer\Service\StructureAnalyzer;

class ProjectDfdCommands extends DrushCommands {

  /** @var \Drupal\drupal_site_dfd_analyzer\Service\StructureAnalyzer */
  protected $analyzer;

  public function __construct(StructureAnalyzer $analyzer) {
    parent::__construct();
    $this->analyzer = $analyzer;
  }

  /**
   * Export project structure as JSON, Mermaid, or Draw.io XML.
   *
   * @command project-dfd:export
   * @option json Path to write JSON file
   * @option mermaid Path to write Mermaid file
   * @option drawio Path to write Draw.io XML file
   * @usage drush project-dfd:export --json=/tmp/dfd.json --mermaid=/tmp/dfd.mmd --drawio=/tmp/dfd.drawio
   */
  public function export(array $options = ['json' => NULL, 'mermaid' => NULL, 'drawio' => NULL]) {
    $data = $this->analyzer->analyze();
    $mermaid = $this->analyzer->buildMermaid($data);

    if (!empty($options['json'])) {
      file_put_contents($options['json'], json_encode($data, JSON_PRETTY_PRINT));
      $this->logger()->success(dt('Wrote JSON to @path', ['@path' => $options['json']]));
    }
    if (!empty($options['mermaid'])) {
      file_put_contents($options['mermaid'], $mermaid);
      $this->logger()->success(dt('Wrote Mermaid to @path', ['@path' => $options['mermaid']]));
    }
    if (!empty($options['drawio'])) {
      $drawio = $this->buildDrawioXml($data);
      file_put_contents($options['drawio'], $drawio);
      $this->logger()->success(dt('Wrote Draw.io XML to @path', ['@path' => $options['drawio']]));
    }

    if (empty($options['json']) && empty($options['mermaid']) && empty($options['drawio'])) {
      $this->output()->writeln(json_encode(['json' => $data, 'mermaid' => $mermaid], JSON_PRETTY_PRINT));
    }
  }

  /**
   * Build Draw.io XML from analyzed data.
   */
  protected function buildDrawioXml(array $data): string {
    $xml = '<?xml version="1.0" encoding="UTF-8"?>
<mxfile host="app.diagrams.net" modified="' . date('c') . '" agent="Drupal Project DFD Analyzer" version="22.1.16" etag="' . uniqid() . '" type="device">
  <diagram name="Project DFD" id="' . uniqid() . '">
    <mxGraphModel dx="1422" dy="794" grid="1" gridSize="10" guides="1" tooltips="1" connect="1" arrows="1" fold="1" page="1" pageScale="1" pageWidth="1169" pageHeight="827" math="0" shadow="0">
      <root>
        <mxCell id="0" />
        <mxCell id="1" parent="0" />
        <mxCell id="2" value="Project Data Flow Diagram" style="text;html=1;strokeColor=none;fillColor=none;align=center;verticalAlign=middle;whiteSpace=wrap;rounded=0;fontSize=24;fontStyle=1" vertex="1" parent="1">
          <mxGeometry x="400" y="20" width="300" height="40" as="geometry" />
        </mxCell>';

    $y = 100;
    $x = 50;
    $nodeId = 10;

    // Add content types
    foreach ($data['bundles'] as $bundleId => $bundle) {
      $xml .= '
        <mxCell id="' . $nodeId . '" value="' . htmlspecialchars($bundle['label']) . '&#xa;(' . strtoupper($bundle['entity_type']) . ')" style="rounded=1;whiteSpace=wrap;html=1;fillColor=#dae8fc;strokeColor=#6c8ebf;fontStyle=1" vertex="1" parent="1">
          <mxGeometry x="' . $x . '" y="' . $y . '" width="120" height="60" as="geometry" />
        </mxCell>';
      $nodeId++;
      $x += 150;
      if ($x > 1000) {
        $x = 50;
        $y += 100;
      }
    }

    // Add paragraphs
    foreach ($data['paragraphs'] as $bundleId => $bundle) {
      $xml .= '
        <mxCell id="' . $nodeId . '" value="' . htmlspecialchars($bundle['label']) . '&#xa;(PARAGRAPH)" style="rounded=1;whiteSpace=wrap;html=1;fillColor=#d5e8d4;strokeColor=#82b366;fontStyle=1" vertex="1" parent="1">
          <mxGeometry x="' . $x . '" y="' . $y . '" width="120" height="60" as="geometry" />
        </mxCell>';
      $nodeId++;
      $x += 150;
      if ($x > 1000) {
        $x = 50;
        $y += 100;
      }
    }

    // Add taxonomy
    foreach ($data['taxonomies'] as $bundleId => $bundle) {
      $xml .= '
        <mxCell id="' . $nodeId . '" value="' . htmlspecialchars($bundle['label']) . '&#xa;(TAXONOMY)" style="ellipse;whiteSpace=wrap;html=1;fillColor=#fff2cc;strokeColor=#d6b656;fontStyle=1" vertex="1" parent="1">
          <mxGeometry x="' . $x . '" y="' . $y . '" width="120" height="60" as="geometry" />
        </mxCell>';
      $nodeId++;
      $x += 150;
      if ($x > 1000) {
        $x = 50;
        $y += 100;
      }
    }

    $xml .= '
      </root>
    </mxGraphModel>
  </diagram>
</mxfile>';

    return $xml;
  }
}
