<?php

namespace Drupal\drupal_site_dfd_analyzer\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Collects site structure data: content types, fields, paragraphs, taxonomy, views.
 */
class StructureAnalyzer {

  /** @var \Drupal\Core\Entity\EntityTypeManagerInterface */
  protected $entityTypeManager;

  /** @var \Drupal\Core\Entity\EntityFieldManagerInterface */
  protected $entityFieldManager;

  /** @var \Drupal\Core\Config\ConfigFactoryInterface */
  protected $configFactory;

  /**
   * StructureAnalyzer constructor.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    EntityFieldManagerInterface $entityFieldManager,
    ConfigFactoryInterface $configFactory
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
    $this->configFactory = $configFactory;
  }

  /**
   * Build a structured array describing entities and relationships.
   */
  public function analyze(): array {
    $data = [
      'bundles' => [],
      'relationships' => [],
      'views' => [],
      'taxonomies' => [],
      'paragraphs' => [],
      'media' => [],
      'webforms' => [],
      'meta' => [
        'generated' => date('c'),
        'site' => \Drupal::config('system.site')->get('name'),
      ],
    ];

    // Nodes: content types and their fields.
    $nodeTypes = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    foreach ($nodeTypes as $type) {
      $bundleId = 'node.' . $type->id();
      $data['bundles'][$bundleId] = [
        'entity_type' => 'node',
        'bundle' => $type->id(),
        'label' => $type->label(),
        'fields' => [],
      ];
      $fields = $this->entityFieldManager->getFieldDefinitions('node', $type->id());
      foreach ($fields as $fieldName => $definition) {
        // Skip system/base fields that clutter the diagram
        if ($this->isSystemField($fieldName)) {
          continue;
        }
        
        $fieldType = $definition->getType();
        $data['bundles'][$bundleId]['fields'][$fieldName] = [
          'type' => $fieldType,
          'target' => $definition->getSetting('target_type') ?? null,
          'handler' => $definition->getSetting('handler') ?? null,
        ];
        if ($fieldType === 'entity_reference' || $fieldType === 'entity_reference_revisions') {
          $targetType = (string) $definition->getSetting('target_type');
          $handlerSettings = (array) $definition->getSetting('handler_settings');
          $targetBundles = $handlerSettings['target_bundles'] ?? [];
          $data['relationships'][] = [
            'source' => $bundleId,
            'field' => $fieldName,
            'type' => $fieldType,
            'target_entity_type' => $targetType,
            'target_bundles' => array_values($targetBundles),
          ];
        }
      }
    }

    // Taxonomy vocabularies.
    $vocabularies = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();
    foreach ($vocabularies as $vocabulary) {
      $bundleId = 'taxonomy_term.' . $vocabulary->id();
      $data['taxonomies'][$bundleId] = [
        'entity_type' => 'taxonomy_term',
        'bundle' => $vocabulary->id(),
        'label' => $vocabulary->label(),
      ];
    }

    // Paragraph types (if module enabled).
    if ($this->entityTypeManager->hasDefinition('paragraphs_type')) {
      $paragraphTypes = $this->entityTypeManager->getStorage('paragraphs_type')->loadMultiple();
      foreach ($paragraphTypes as $pType) {
        $bundleId = 'paragraph.' . $pType->id();
        $data['paragraphs'][$bundleId] = [
          'entity_type' => 'paragraph',
          'bundle' => $pType->id(),
          'label' => $pType->label(),
          'fields' => [],
        ];
        $fields = $this->entityFieldManager->getFieldDefinitions('paragraph', $pType->id());
        foreach ($fields as $fieldName => $definition) {
          // Skip system/base fields that clutter the diagram
          if ($this->isSystemField($fieldName)) {
            continue;
          }
          
          $fieldType = $definition->getType();
          $data['paragraphs'][$bundleId]['fields'][$fieldName] = [
            'type' => $fieldType,
            'target' => $definition->getSetting('target_type') ?? null,
            'handler' => $definition->getSetting('handler') ?? null,
          ];
          if ($fieldType === 'entity_reference' || $fieldType === 'entity_reference_revisions') {
            $targetType = (string) $definition->getSetting('target_type');
            $handlerSettings = (array) $definition->getSetting('handler_settings');
            $targetBundles = $handlerSettings['target_bundles'] ?? [];
            $data['relationships'][] = [
              'source' => $bundleId,
              'field' => $fieldName,
              'type' => $fieldType,
              'target_entity_type' => $targetType,
              'target_bundles' => array_values($targetBundles),
            ];
          }
        }
      }
    }

    // Media types (if module enabled).
    if ($this->entityTypeManager->hasDefinition('media_type')) {
      $mediaTypes = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
      foreach ($mediaTypes as $mediaType) {
        $bundleId = 'media.' . $mediaType->id();
        $data['media'][$bundleId] = [
          'entity_type' => 'media',
          'bundle' => $mediaType->id(),
          'label' => $mediaType->label(),
          'fields' => [],
        ];
        $fields = $this->entityFieldManager->getFieldDefinitions('media', $mediaType->id());
        foreach ($fields as $fieldName => $definition) {
          // Skip system/base fields that clutter the diagram
          if ($this->isSystemField($fieldName)) {
            continue;
          }
          
          $fieldType = $definition->getType();
          $data['media'][$bundleId]['fields'][$fieldName] = [
            'type' => $fieldType,
            'target' => $definition->getSetting('target_type') ?? null,
            'handler' => $definition->getSetting('handler') ?? null,
          ];
          if ($fieldType === 'entity_reference' || $fieldType === 'entity_reference_revisions') {
            $targetType = (string) $definition->getSetting('target_type');
            $handlerSettings = (array) $definition->getSetting('handler_settings');
            $targetBundles = $handlerSettings['target_bundles'] ?? [];
            $data['relationships'][] = [
              'source' => $bundleId,
              'field' => $fieldName,
              'type' => $fieldType,
              'target_entity_type' => $targetType,
              'target_bundles' => array_values($targetBundles),
            ];
          }
        }
      }
    }

    // Webforms (if module enabled).
    if ($this->entityTypeManager->hasDefinition('webform')) {
      $webforms = $this->entityTypeManager->getStorage('webform')->loadMultiple();
      foreach ($webforms as $webform) {
        $bundleId = 'webform.' . $webform->id();
        $elements = $webform->getElementsDecoded();
        $fields = [];
        
        // Extract form elements as "fields"
        foreach ($elements as $key => $element) {
          $fields[$key] = [
            'type' => $element['#type'] ?? 'textfield',
            'title' => $element['#title'] ?? $key,
          ];
        }
        
        $data['webforms'][$bundleId] = [
          'entity_type' => 'webform',
          'bundle' => $webform->id(),
          'label' => $webform->label(),
          'fields' => $fields,
        ];
      }
    }

    // Views metadata.
    $viewsStorage = $this->entityTypeManager->getStorage('view');
    if ($viewsStorage) {
      $views = $viewsStorage->loadMultiple();
      foreach ($views as $view) {
        $data['views'][] = [
          'id' => $view->id(),
          'label' => $view->label(),
          'base_table' => $view->get('base_table'),
        ];
      }
    }

    return $data;
  }

  /**
   * Build a Mermaid graph from the analyzed data.
   */
  public function buildMermaid(array $data): string {
    $lines = [];
    $lines[] = 'flowchart TD';

    // Nodes for bundles.
    foreach ($data['bundles'] as $bundleId => $bundle) {
      $lines[] = sprintf('%s["%s %s"]', $this->sanitizeId($bundleId), strtoupper($bundle['entity_type']), $bundle['bundle']);
    }
    // Paragraph bundles.
    foreach ($data['paragraphs'] as $bundleId => $bundle) {
      $lines[] = sprintf('%s(["%s %s"])', $this->sanitizeId($bundleId), strtoupper($bundle['entity_type']), $bundle['bundle']);
    }
    // Taxonomy vocabularies.
    foreach ($data['taxonomies'] as $bundleId => $bundle) {
      $lines[] = sprintf('%s{{"%s %s"}}', $this->sanitizeId($bundleId), 'TAXONOMY', $bundle['bundle']);
    }

    // For relationships, ensure target nodes exist at least as generic nodes.
    $definedNodes = [];
    foreach (array_merge(array_keys($data['bundles']), array_keys($data['paragraphs']), array_keys($data['taxonomies'])) as $id) {
      $definedNodes[$this->sanitizeId($id)] = TRUE;
    }

    foreach ($data['relationships'] as $rel) {
      $source = $this->sanitizeId($rel['source']);
      $label = $rel['field'];
      if (!empty($rel['target_bundles'])) {
        foreach ($rel['target_bundles'] as $targetBundle) {
          $rawTarget = $rel['target_entity_type'] . '.' . $targetBundle;
          $targetId = $this->sanitizeId($rawTarget);
          if (!isset($definedNodes[$targetId])) {
            $lines[] = sprintf('%s["%s"]', $targetId, strtoupper($rel['target_entity_type']) . ' ' . $targetBundle);
            $definedNodes[$targetId] = TRUE;
          }
          $lines[] = sprintf('%s -- "%s" --> %s', $source, $this->escapeLabel($label), $targetId);
        }
      }
      else {
        $rawTarget = $rel['target_entity_type'] ?? 'entity';
        $targetId = $this->sanitizeId($rawTarget);
        if (!isset($definedNodes[$targetId])) {
          $lines[] = sprintf('%s["%s"]', $targetId, strtoupper($rawTarget));
          $definedNodes[$targetId] = TRUE;
        }
        $lines[] = sprintf('%s -- "%s" --> %s', $source, $this->escapeLabel($label), $targetId);
      }
    }

    return implode("\n", $lines);
  }

  protected function sanitizeId(string $id): string {
    return preg_replace('/[^A-Za-z0-9_]/', '_', $id);
  }

  protected function escapeLabel(string $label): string {
    return str_replace(['"'], ['\\"'], $label);
  }

  /**
   * Check if a field is a system field that should be excluded from the diagram.
   */
  protected function isSystemField(string $fieldName): bool {
    $systemFields = [
      // Entity base fields
      'nid', 'vid', 'uuid', 'langcode', 'type', 'status', 'uid', 'created', 'changed',
      'promote', 'sticky', 'default_langcode', 'revision_timestamp', 'revision_uid',
      'revision_log', 'revision_default', 'revision_translation_affected',
      
      // Media base fields
      'mid', 'bundle', 'name', 'thumbnail',
      
      // Paragraph base fields
      'id', 'parent_id', 'parent_type', 'parent_field_name', 'behavior_settings',
      
      // Common system fields
      'path', 'menu_link', 'comment', 'moderation_state',
      
      // Layout builder and other system fields
      'layout_builder__layout', 'field_meta_tags',
      
      // Skip very basic fields that don't add value to the diagram
      'title' // Keep title for now, but can be excluded if needed
    ];
    
    return in_array($fieldName, $systemFields);
  }
}
