<?php

declare(strict_types=1);

namespace Drupal\drupalauth4ssp\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\drupalauth4ssp\SspHandler;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Returns responses for DrupalAuth for SimpleSAMLphp routes.
 */
final class RedirectController extends ControllerBase {

  /**
   * Parameter name to store redirect destination in the session.
   */
  public const SESSION_PARAM = 'drupalauth4ssp.ReturnTo';

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly SessionInterface $session,
    private readonly SspHandler $sspHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('session'),
      $container->get('drupalauth4ssp.ssp_handler'),
    );
  }

  /**
   * Redirects the user back to the service provider they came from.
   */
  public function redirectToServiceProvider(): RedirectResponse {
    $returnTo = $this->session->get(self::SESSION_PARAM);
    if (!empty($returnTo) && $this->sspHandler->returnPathIsAllowed($returnTo)) {
      $this->session->remove(self::SESSION_PARAM);
      return new TrustedRedirectResponse($returnTo);
    }
    else {
      // Using \Drupal\Core\Routing\LocalRedirectResponse makes more sense here,
      // but that throws LogicException.
      // See https://www.drupal.org/project/drupal/issues/3463642
      return new RedirectResponse(
        Url::fromRoute('<front>')->toString()
      );
    }

  }

}
