<?php

namespace Drupal\drupalauth4ssp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use SimpleSAML\Utils\HTTP;

/**
 * Configure DrupalAuth for SimpleSAMLphp settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'drupalauth4ssp_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int,string>
   *   An array of configuration object names that are editable.
   */
  protected function getEditableConfigNames() {
    return ['drupalauth4ssp.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.iterableValue (Ignore $form variable)
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['returnto_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed list of URLs for ReturnTo Parameter for service providers'),
      '#default_value' => implode(PHP_EOL, $this
        ->config('drupalauth4ssp.settings')
        ->get('returnto_list')),
      '#description' => $this->t('Enter one URL per line. The "*"(wildcard) character is allowed. Example URLs are www.example.com/specific-path for a certain path and www.example.com* for all the URLs for www.example.com domain (like www.example.com; www.example.com/path1; www.example.com/path2/path3 etc.) and *example.com* for all subdomain paths (like a.example.com; a.example.com/path etc.)'),
      '#required' => TRUE,
    ];

    $form['idp_logout_returnto'] = [
      '#type' => 'textfield',
      '#title' => $this->t('IdP-initiated logout redirect URL'),
      '#default_value' => $this
        ->config('drupalauth4ssp.settings')
        ->get('idp_logout_returnto'),
      '#description' => $this->t('URL where to return the user after SimpleSAMLphp will finish logout process. Leave empty to return to the home page.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.iterableValue (Ignore $form variable)
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $returnTo = $form_state->getValue('idp_logout_returnto');
    try {
      $http = new HTTP();
      $http->checkURLAllowed($returnTo);
    }
    catch (\Exception $exception) {
      $form_state->setErrorByName('idp_logout_returnto', $exception->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore missingType.iterableValue (Ignore $form variable)
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $returnToList = array_map('trim', explode(PHP_EOL, $form_state->getValue('returnto_list')));
    $this->config('drupalauth4ssp.settings')
      ->set('returnto_list', $returnToList)
      ->set('idp_logout_returnto', $form_state->getValue('idp_logout_returnto'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
