/**
 * @file
 * Drupal Fit security report behaviors.
 */

(function (Drupal) {
  'use strict';

  /**
   * Behavior for Drupal Fit security report interactions.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.drupalfitReport = {
    attach: function (context, settings) {
      // Helper function to get color based on score.
      const getScoreColor = (score) => {
        const colors = settings.drupalfit?.scoreColors || [
          {threshold: 80, color: '#4caf50'},
          {threshold: 60, color: '#ffc107'},
          {threshold: 40, color: '#ff9800'},
          {threshold: 0, color: '#dc3545'}
        ];
        
        for (const item of colors) {
          if (score >= item.threshold) {
            return item.color;
          }
        }
        return colors[colors.length - 1].color;
      };

      // Animate score circles.
      const circles = context.querySelectorAll('.score-circle__progress');
      circles.forEach((circle) => {
        if (!circle.dataset.animated) {
          const score = parseInt(circle.dataset.score, 10);
          const circumference = 282.7; // 2 * PI * 45
          const offset = circumference - (score / 100) * circumference;
          circle.style.stroke = getScoreColor(score);
          // Animate the circle.
          setTimeout(() => {
            circle.style.strokeDasharray = `${circumference - offset} ${circumference}`;
          }, 100);
          circle.dataset.animated = 'true';
        }
      });

      // Animate group progress bars.
      const progressBars = context.querySelectorAll('.group-score-item__progress');
      progressBars.forEach((bar) => {
        if (!bar.dataset.animated) {
          const score = parseInt(bar.dataset.score, 10);
          const color = getScoreColor(score);
          bar.style.backgroundColor = color;
          bar.style.width = '0%';
          setTimeout(() => {
            // Show at least 2% width for visibility when score is very low
            const displayWidth = score === 0 ? 2 : score;
            bar.style.width = `${displayWidth}%`;
          }, 100);
          bar.dataset.animated = 'true';
        }
      });

      // Connect group score items to their report sections.
      const groupItems = context.querySelectorAll('.group-score-item');
      groupItems.forEach((item) => {
        if (!item.dataset.connected) {
          item.style.cursor = 'pointer';
          item.addEventListener('click', function() {
            const groupName = this.querySelector('.group-score-item__name').textContent.trim().toLowerCase().replace(/ /g, '_');
            const groupSection = document.querySelector(`.drupalfit-group[data-group="${groupName}"]`);
            if (groupSection) {
              groupSection.open = true;
              groupSection.scrollIntoView({behavior: 'smooth', block: 'start'});
            }
          });
          item.dataset.connected = 'true';
        }
      });
    }
  };

})(Drupal);
