<?php

declare(strict_types=1);

namespace Drupal\drupalfit;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Enum\FitWeight;

/**
 * Represents a fitness check result with messages and weight.
 */
class FitResult {

  public function __construct(
    public readonly string $fitId,
    public readonly string $name,
    public readonly string $fitGroup,
    public FitWeight $weight,
    public ?TranslatableMarkup $description = NULL,
    public ?TranslatableMarkup $successMessage = NULL,
    public ?TranslatableMarkup $failureMessage = NULL,
    public ?TranslatableMarkup $warningMessage = NULL,
    public ?TranslatableMarkup $errorMessage = NULL,
    public ?TranslatableMarkup $infoMessage = NULL,
    public array $helpMessage = [],
  ) {}

  /**
   * Creates a new FitResult instance.
   */
  public static function create(
    string $fitId,
    string $name,
    string $fitGroup,
    FitWeight $weight,
    ?TranslatableMarkup $description = NULL,
    ?TranslatableMarkup $successMessage = NULL,
    ?TranslatableMarkup $failureMessage = NULL,
    ?TranslatableMarkup $warningMessage = NULL,
    ?TranslatableMarkup $errorMessage = NULL,
    ?TranslatableMarkup $infoMessage = NULL,
    array $helpMessage = [],
  ) : static {
    return new static(
      $fitId,
      $name,
      $fitGroup,
      $weight,
      $description,
      $successMessage,
      $failureMessage,
      $warningMessage,
      $errorMessage,
      $infoMessage,
      $helpMessage
    );
  }

  /**
   * Gets the fit ID.
   */
  public function id(): string {
    return $this->fitId;
  }

  /**
   * Gets the fit name.
   */
  public function name(): string {
    return $this->name;
  }

  /**
   * Gets the fit group.
   */
  public function group(): string {
    return $this->fitGroup;
  }

  /**
   * Gets the fit weight.
   */
  public function weight(): FitWeight {
    return $this->weight;
  }

  /**
   * Sets the success message.
   */
  public function setSuccessMessage(?TranslatableMarkup $successMessage): static {
    $this->successMessage = $successMessage;
    return $this;
  }

  /**
   * Sets the failure message.
   */
  public function setFailureMessage(?TranslatableMarkup $failureMessage): static {
    $this->failureMessage = $failureMessage;
    return $this;
  }

  /**
   * Sets the warning message.
   */
  public function setWarningMessage(?TranslatableMarkup $warningMessage): static {
    $this->warningMessage = $warningMessage;
    return $this;
  }

  /**
   * Sets the error message.
   */
  public function setErrorMessage(?TranslatableMarkup $errorMessage): static {
    $this->errorMessage = $errorMessage;
    return $this;
  }

  /**
   * Sets the info message.
   */
  public function setInfoMessage(?TranslatableMarkup $infoMessage): static {
    $this->infoMessage = $infoMessage;
    return $this;
  }

  /**
   * Adds a help message.
   */
  public function setHelpMessage(TranslatableMarkup|array $helpMessage): static {
    $this->helpMessage[] = is_array($helpMessage) ? $helpMessage : ['#markup' => $helpMessage];
    return $this;
  }

  /**
   * Gets help render array.
   */
  public function getHelpRenderArray(): ?array {
    return empty($this->helpMessage) ? NULL : $this->helpMessage;
  }

  /**
   * Gets the success message.
   */
  public function getSuccessMessage(): ?TranslatableMarkup {
    return $this->successMessage;
  }

  /**
   * Gets the failure message.
   */
  public function getFailureMessage(): ?TranslatableMarkup {
    return $this->failureMessage;
  }

  /**
   * Gets the warning message.
   */
  public function getWarningMessage(): ?TranslatableMarkup {
    return $this->warningMessage;
  }

  /**
   * Gets the error message.
   */
  public function getErrorMessage(): ?TranslatableMarkup {
    return $this->errorMessage;
  }

  /**
   * Gets the info message.
   */
  public function getInfoMessage(): ?TranslatableMarkup {
    return $this->infoMessage;
  }

  /**
   * Gets the help messages.
   */
  public function getHelpMessage(): array {
    return $this->helpMessage;
  }

  /**
   * Sets the weight.
   */
  public function setWeight(FitWeight $weight): static {
    $this->weight = $weight;
    return $this;
  }

  /**
   * Gets the description.
   */
  public function getDescription(): ?TranslatableMarkup {
    return $this->description;
  }

  /**
   * Converts to array.
   */
  public function toArray() : array {
    return [
      'id' => $this->id(),
      'name' => $this->name(),
      'group' => $this->group(),
      'description' => $this->getDescription()?->render(),
      'weightName' => $this->weight()->name,
      'weight' => $this->weight()->value,
      'successMessage' => $this->getSuccessMessage()?->render(),
      'failureMessage' => $this->getFailureMessage()?->render(),
      'warningMessage' => $this->getWarningMessage()?->render(),
      'errorMessage' => $this->getErrorMessage()?->render(),
      'infoMessage' => $this->getInfoMessage()?->render(),
      'helpMessage' => array_map(static fn($message) => $message instanceof TranslatableMarkup ? $message->render() : $message, $this->getHelpMessage()),
      'helpRenderArray' => $this->getHelpRenderArray(),
    ];
  }

}
