<?php

declare(strict_types=1);

namespace Drupal\drupalfit;

use Drupal\drupalfit\Enum\FitWeight;

/**
 * Collection of FitResult objects.
 */
class FitResultCollection implements \Iterator, \Countable {

  /**
   * Array of FitResult objects.
   *
   * @var \Drupal\drupalfit\FitResult[]
   */
  public array $results = [];

  /**
   * Current position for iterator.
   *
   * @var int
   */
  private int $position = 0;

  /**
   * Adds a result.
   */
  public function add(FitResult $result): void {
    $this->results[] = $result;
  }

  /**
   * Gets all results.
   */
  public function getAll(): array {
    return $this->results;
  }

  /**
   * Gets results by group.
   */
  public function getByGroup(string $group): array {
    return array_filter($this->results, static fn($result) => $result->group() === $group);
  }

  /**
   * Gets results by weight.
   */
  public function getByWeight(FitWeight $weight): array {
    return array_filter($this->results, fn($result) => $result->weight() === $weight);
  }

  /**
   * Gets the highest severity weight.
   */
  public function getHighestSeverity(): ?FitWeight {
    if (empty($this->results)) {
      return NULL;
    }

    $weights = array_map(static fn($result) => $result->weight(), $this->results);
    return min($weights);
  }

  /**
   * Counts results by weight.
   */
  public function countByWeight(): array {
    $counts = [];
    foreach (FitWeight::cases() as $weight) {
      $counts[$weight->name] = count($this->getByWeight($weight));
    }
    return $counts;
  }

  /**
   * {@inheritdoc}
   */
  public function current(): FitResult {
    return $this->results[$this->position];
  }

  /**
   * {@inheritdoc}
   */
  public function key(): int {
    return $this->position;
  }

  /**
   * {@inheritdoc}
   */
  public function next(): void {
    ++$this->position;
  }

  /**
   * {@inheritdoc}
   */
  public function rewind(): void {
    $this->position = 0;
  }

  /**
   * {@inheritdoc}
   */
  public function valid(): bool {
    return isset($this->results[$this->position]);
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return count($this->results);
  }

  /**
   * Checks if collection is empty.
   */
  public function isEmpty(): bool {
    return empty($this->results);
  }

  /**
   * Clears all results.
   */
  public function clear(): void {
    $this->results = [];
    $this->position = 0;
  }

  /**
   * Gets first result.
   */
  public function first(): ?FitResult {
    return $this->results[0] ?? NULL;
  }

  /**
   * Gets last result.
   */
  public function last(): ?FitResult {
    return end($this->results) ?: NULL;
  }

  /**
   * Checks if result exists by ID.
   */
  public function has(string $id): bool {
    return $this->get($id) !== NULL;
  }

  /**
   * Gets result by ID.
   */
  public function get(string $id): ?FitResult {
    foreach ($this->results as $result) {
      if ($result->id() === $id) {
        return $result;
      }
    }
    return NULL;
  }

  /**
   * Removes result by ID.
   */
  public function remove(string $id): bool {
    foreach ($this->results as $key => $result) {
      if ($result->id() === $id) {
        unset($this->results[$key]);
        $this->results = array_values($this->results);
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Filters results by callback.
   */
  public function filter(callable $callback): array {
    return array_filter($this->results, $callback);
  }

  /**
   * Maps results using callback.
   */
  public function map(callable $callback): array {
    return array_map($callback, $this->results);
  }

  /**
   * Converts to array.
   */
  public function toArray(): array {
    return array_map(static fn($result) => $result->toArray(), $this->results);
  }

}
