<?php

declare(strict_types=1);

namespace Drupal\drupalfit;

use Drupal\Core\Url;

/**
 * Groups fit results for display.
 */
class FitResultGrouper {

  /**
   * Groups results by group and severity.
   */
  public static function groupBySeverity(FitResultCollection $results, array $groupWeights = []): array {
    $grouped = [];

    foreach ($results->getAll() as $result) {
      $group = $result->group();
      $severity = $result->weight()->name;

      $grouped[$group][$severity][] = [
        'id' => $result->id(),
        'name' => $result->name(),
        'description' => $result->getDescription(),
        'weight' => $result->weight(),
        'success_message' => $result->getSuccessMessage(),
        'failure_message' => $result->getFailureMessage(),
        'warning_message' => $result->getWarningMessage(),
        'error_message' => $result->getErrorMessage(),
        'info_message' => $result->getInfoMessage(),
        'help_messages' => $result->getHelpMessage(),
        'help_render_array' => $result->getHelpRenderArray(),
      ];
    }

    $sortedGrouped = [];
    $groupsWithWeights = [];
    foreach (array_keys($grouped) as $group) {
      $groupsWithWeights[$group] = $groupWeights[$group] ?? 0;
    }
    asort($groupsWithWeights);

    foreach (array_keys($groupsWithWeights) as $group) {
      $severities = $grouped[$group];
      uksort($severities, function ($a, $b) {
        $weightA = constant("Drupal\\drupalfit\\Enum\\FitWeight::$a")->value;
        $weightB = constant("Drupal\\drupalfit\\Enum\\FitWeight::$b")->value;
        return $weightA <=> $weightB;
      });
      $sortedGrouped[$group] = $severities;
    }

    return $sortedGrouped;
  }

  /**
   * Prepares group scores for display.
   */
  public static function prepareGroupScores(GroupScoreCollection $groupScores, array $groupWeights = []): array {
    $scores = [];

    foreach ($groupScores->getScores() as $groupScore) {
      $groupName = $groupScore->getGroup();
      $scores[$groupName] = [
        'score' => $groupScore->getScore(),
        'max_score' => FitScoreCalculator::MAX_SCORE,
        'external_provider' => $groupScore->isExternalProvider(),
        'external_url' => $groupScore->isExternalProvider() ? Url::fromRoute('drupalfit.drupalfit_report')->toString() : NULL,
        'weight' => $groupWeights[$groupName] ?? 0,
      ];
    }
    uasort($scores, static fn($a, $b) => $a['weight'] <=> $b['weight']);
    return $scores;
  }

}
