<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'cache_check',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('Cache Configuration'),
  description: new TranslatableMarkup('Checks if caching is properly configured.'),
  successMessage: new TranslatableMarkup('Caching is properly configured.'),
  failureMessage: new TranslatableMarkup('Caching configuration needs improvement.'),
)]
class Cache extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $config = $this->configFactory->get('system.performance');
    $pageMaxAge = $config->get('cache.page.max_age');
    $dynamicPageCache = $this->moduleHandler->moduleExists('dynamic_page_cache');
    $pageCache = $this->moduleHandler->moduleExists('page_cache');
    $issues = [];

    if (!$pageCache) {
      $issues[] = 'Page Cache module is disabled';
    }
    elseif ($pageMaxAge === 0) {
      $issues[] = 'Page cache max age is 0';
    }
    elseif ($pageMaxAge < 900) {
      $issues[] = "Page cache max age is low ({$pageMaxAge}s, recommended: 900s+)";
    }

    if (!$dynamicPageCache) {
      $issues[] = 'Dynamic Page Cache module is disabled';
    }

    if (!empty($issues)) {
      $weight = (!$pageCache || $pageMaxAge === 0) ? FitWeight::High : FitWeight::Medium;
      $result
        ->setWeight($weight)
        ->setFailureMessage($this->t('Issues found: @issues', ['@issues' => implode('; ', $issues)]))
        ->setHelpMessage($this->t('Enable Page Cache and Dynamic Page Cache modules.'))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Performance settings'),
          '#url' => Url::fromRoute('system.performance_settings'),
        ])
        ->setHelpMessage($this->t('Set page cache max age to at least 15 minutes (900 seconds).'));
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
