<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Password\PhpPassword;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'username_password_match_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('Weak Passwords'),
  description: new TranslatableMarkup('Checks for weak password and users with username matching password.'),
  successMessage: new TranslatableMarkup('No weak password patterns detected.'),
  failureMessage: new TranslatableMarkup('Weak passwords detected.'),
)]
class CheckUsernamePasswordMatch extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly PhpPassword $password,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('password')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $userStorage = $this->entityTypeManager
      ->getStorage('user');

    /** @var \Drupal\user\UserInterface $users */
    $users = $userStorage->loadMultiple();
    $weakUsers = [];

    foreach ($users as $user) {
      if ($user->id() === 0) {
        continue;
      }

      $username = $user->getAccountName();
      $passHash = $user->getPassword();

      if ($passHash && $this->hasWeakPassword($username, $passHash)) {
        $weakUsers[] = $username;
      }
    }

    if (!empty($weakUsers)) {
      $count = count($weakUsers);
      $result
        ->setWeight(FitWeight::Critical)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('@count user(s) may have weak passwords.', ['@count' => $count]))
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ message }} {{ link }}',
          '#context' => [
            'message' => $this->t('Force password reset or enable password policy module.'),
            'link' => [
              '#type' => 'link',
              '#title' => $this->t('Manage users'),
              '#url' => Url::fromRoute('entity.user.collection'),
            ],
          ],
        ]);

      foreach (array_slice($weakUsers, 0, 3) as $username) {
        $result->setHelpMessage($this->t('- @username', ['@username' => $username]));
      }

      if ($count > 3) {
        $result->setHelpMessage($this->t('...and @more more', ['@more' => $count - 3]));
      }
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

  /**
   * Check if user has weak password.
   */
  private function hasWeakPassword(string $username, string $passHash): bool {
    foreach ($this->weakPasswordSet($username) as $weak) {
      if ($this->password->check($weak, $passHash)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Command weak password set with the username.
   */
  protected function weakPasswordSet($userName) : array {
    return [
      $userName,
      // Common dictionary words.
      'password',
      'Password',
      'password123',
      'password1',
      'password12',
      'password1234',
      'password12345',
      'p@ssw0rd',
      'iloveyou',
      'welcome',
      'secret',
      'letmein',
      'sunshine',
      'dragon',
      'monkey',
      'football',
      'baseball',
      'princess',
      'superman',
      'shadow',
      'starwars',
      'liverpool',
      'master',
      'lovely',
      'charlie',
      'donald',

      // Admin and system related.
      'admin',
      'Admin',
      'admin123',
      'admin2025',
      'root',
      'user',
      'guest',
      'test',
      'demo',
      'login',
      'service',
      'manager',
      'security',
      'internet',
      'example',
      'changeme',

      // Drupal specific.
      'drupal',
      'Drupal',
      'drupal123',
      'site',
      'content',
      'node',
      'user1',
      'testuser',
      'developer',
      'staging',
      'localhost',
      'default',

      // Numeric sequences.
      '123456',
      '123456789',
      '12345678',
      '12345',
      '1234567',
      '1234567890',
      '1234',
      '123',
      '111111',
      '222222',
      '000000',
      '555555',
      '666666',
      '777777',
      '7777777',
      '888888',
      '654321',
      '102030',
      '123123',
      '123321',

      // Keyboard patterns.
      'qwerty',
      'qwerty123',
      'qwertyuiop',
      'asdfgh',
      'abc123',
      '1q2w3e',
      '1q2w3e4r',
      '123qwe',
      'qqww1122',

      // Year-based.
      '2025',
      '2024',
      'password2025',

      // Other common weak passwords.
      'picture1',
      'senha',
      'Million2',
      'OOOOOO',
      'aaron431',
      'omgpop',
      '!@#$%^&*',
      'aa123456',
    ];
  }

}
