<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Drupal\update\UpdateManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Checks for contributed module updates.
 */
#[FitCheck(
  id: 'contrib_update_check',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Contributed Modules'),
  description: new TranslatableMarkup('Checks for available updates to contributed modules.'),
  successMessage: new TranslatableMarkup('All modules are up to date.'),
  failureMessage: new TranslatableMarkup('Module updates available.'),
)]
class ContribUpdateCheck extends FitCheckPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    if (!$this->moduleHandler->moduleExists('update')) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->t('Update module is disabled.'))
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}',
          '#context' => ['text' => $this->t('Enable the Update Manager module.')],
        ]);
      return $result;
    }

    $available = update_get_available(TRUE);

    if (empty($available)) {
      $result
        ->setWeight(FitWeight::Info)
        ->setInfoMessage($this->t('No update data available.'))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Check for updates'),
          '#url' => Url::fromRoute('update.manual_status'),
        ]);
      return $result;
    }

    $data = update_calculate_project_data($available);
    $outdatedModules = [];
    $hasSecurityIssue = FALSE;

    foreach ($data as $project => $info) {
      if ($project === 'drupal' || ($info['project_type'] !== 'module' &&  $info['project_type'] !== 'theme')) {
        continue;
      }

      if ($info['status'] !== UpdateManagerInterface::CURRENT) {
        $outdatedModules[] = [
          'name' => $info['title'] ?? $project,
          'current' => $info['existing_version'] ?? 'Unknown',
          'recommended' => $info['recommended'] ?? 'Unknown',
          'latest' => $info['latest_version'] ?? 'Unknown',
        ];
        if ($info['status'] === UpdateManagerInterface::NOT_SECURE) {
          $hasSecurityIssue = TRUE;
        }
      }
    }

    if (empty($outdatedModules)) {
      $result->setSuccessMessage($this->successMessage());
      return $result;
    }

    $items = [];
    $displayCount = min(5, count($outdatedModules));
    for ($i = 0; $i < $displayCount; $i++) {
      $module = $outdatedModules[$i];
      if ($module['latest'] !== $module['recommended']) {
        $items[] = $module['name'] . ': Recommended ' . $module['recommended'] . ' (Latest ' . $module['latest'] . ')';
      }
      else {
        $items[] = $module['name'] . ': Update to ' . $module['recommended'];
      }
    }

    if (count($outdatedModules) > 5) {
      $items[] = '...and ' . (count($outdatedModules) - 5) . ' more';
    }

    $result
      ->setWeight($hasSecurityIssue ? FitWeight::High : FitWeight::Medium)
      ->setFailureMessage($this->failureMessage())
      ->setHelpMessage([
        '#type' => 'inline_template',
        '#template' => '<strong>{{ label }}</strong><br>{{ items|raw }}<br><a href="{{ url }}">{{ link }}</a>',
        '#context' => [
          'label' => $this->t('Outdated modules:'),
          'items' => implode('<br>', $items),
          'url' => '/admin/reports/updates',
          'link' => $this->t('View all updates'),
        ],
      ]);

    return $result;
  }

}
