<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'dangerous_permission_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('Role Permissions'),
  description: new TranslatableMarkup('Checks for dangerous permissions assigned to non-admin roles.'),
  successMessage: new TranslatableMarkup('Role permissions are secure.'),
  failureMessage: new TranslatableMarkup('Dangerous permissions assigned.'),
)]
class DangerousPermissionCheck extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $dangerousPerms = $this->predictablePermission();

    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = $this->entityTypeManager
      ->getStorage('user_role')
      ->loadMultiple();
    $issues = FALSE;

    foreach ($roles as $role) {
      if ($role->isAdmin()) {
        continue;
      }

      foreach ($dangerousPerms as $perm) {
        if ($role->hasPermission($perm)) {
          $issues = TRUE;
          $result->setHelpMessage([
            '#type' => 'inline_template',
            '#template' => '{{ message }} {{ link }}',
            '#context' => [
              'message' => $this->t('Remove "@perm" from @role', ['@role' => $role->label(), '@perm' => $perm]),
              'link' => [
                '#type' => 'link',
                '#title' => $this->t('Edit permissions'),
                '#url' => Url::fromRoute('user.admin_permissions'),
              ],
            ],
          ]);
        }
      }
    }

    if ($issues) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage());
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

  /**
   * Gets list of dangerous permissions.
   *
   * @return string[]
   *   Array of permission names.
   */
  public function predictablePermission(): array {
    return [
      'administer site configuration',
      'administer users',
      'administer permissions',
      'administer modules',
      'administer themes',
      'administer software updates',
      'access administration pages',
      'administer content types',
      'administer fields',
      'administer display modes',
      'administer view modes',
      'bypass node access',
      'administer nodes',
      'delete any content',
      'edit any content',
      'administer taxonomy',
      'administer menu',
      'administer blocks',
      'administer views',
      'execute php code',
      'use php for settings',
      'administer filters',
      'administer text formats',
    ];
  }

}
