<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'development_modules_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('Development Modules'),
  description: new TranslatableMarkup('Checks for development modules enabled in production.'),
  successMessage: new TranslatableMarkup('No development modules enabled.'),
  failureMessage: new TranslatableMarkup('Development modules detected.'),
)]
class DevelopmentModules extends FitCheckPluginBase {

  /**
   * Development modules categorized by risk severity.
   */
  protected const array DEV_MODULES = [
    // High Risk - Direct Security Vulnerabilities.
    'devel' => 'High',
    'devel_php' => 'High',
    'devel_generate' => 'High',
    'webprofiler' => 'High',
    'kint' => 'High',
    'update_test' => 'High',
    'examples' => 'High',
    'module_builder' => 'High',
    'rawdebug' => 'High',
    'twig_xdebug' => 'High',

    // Medium Risk - Information Disclosure & Development Tools.
    'stage_file_proxy' => 'Medium',
    'update_ui' => 'Medium',
    'dblog' => 'Medium',
    'search_api_db' => 'Medium',
    'reroute_email' => 'Medium',
    'devel_entity_updates' => 'Medium',
    'config_inspector' => 'Medium',
    'masquerade' => 'Medium',
    'smtp_test' => 'Medium',
    'theme_debug' => 'Medium',
    'config_devel' => 'Medium',

    // Low Risk - UI and Administrative Tools.
    'views_ui' => 'Low',
    'field_ui' => 'Low',
    'config_translation' => 'Low',
    'contextual' => 'Low',
    'shortcut' => 'Low',
    'toolbar' => 'Low',
    'tour' => 'Low',
    'config_update_ui' => 'Low',
    'backup_migrate' => 'Low',

    // Info - Generally Safe But Unnecessary.
    'help' => 'Info',
    'color' => 'Info',
    'statistics' => 'Info',
  ];

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $enabledDevModules = $this->getEnabledDevModules();

    if (empty($enabledDevModules)) {
      $result->setSuccessMessage($this->successMessage());
      return $result;
    }

    $this->buildFailureResult($result, $enabledDevModules);

    return $result;
  }

  /**
   * Get list of enabled development modules.
   *
   * @return array
   *   Array of enabled development modules with their severity levels.
   */
  private function getEnabledDevModules(): array {
    return array_filter(
      self::DEV_MODULES,
      fn($module) => $this->moduleHandler->moduleExists($module),
      ARRAY_FILTER_USE_KEY
    );
  }

  /**
   * Build failure result with module details.
   *
   * @param \Drupal\drupalfit\FitResult $result
   *   The result object to modify.
   * @param array $enabledModules
   *   Array of enabled development modules.
   */
  private function buildFailureResult(FitResult $result, array $enabledModules): void {
    $highestSeverity = $this->getHighestSeverity($enabledModules);
    $groupedModules = $this->groupModulesBySeverity($enabledModules);

    $result
      ->setWeight($highestSeverity)
      ->setFailureMessage($this->failureMessage())
      ->setHelpMessage($this->t('@count development module(s) enabled:', ['@count' => count($enabledModules)]));

    foreach (['High', 'Medium', 'Low', 'Info'] as $severity) {
      if (isset($groupedModules[$severity])) {
        foreach ($groupedModules[$severity] as $module) {
          $result->setHelpMessage($this->t('- @module (@severity)', [
            '@module' => $module,
            '@severity' => $severity,
          ]));
        }
      }
    }

    $result->setHelpMessage([
      '#type' => 'link',
      '#title' => $this->t('Uninstall modules'),
      '#url' => Url::fromRoute('system.modules_uninstall'),
    ]);
  }

  /**
   * Group modules by severity level.
   *
   * @param array $modules
   *   Array of modules with severity levels.
   *
   * @return array
   *   Modules grouped by severity.
   */
  private function groupModulesBySeverity(array $modules): array {
    $grouped = [];
    foreach ($modules as $module => $severity) {
      $grouped[$severity][] = $module;
    }
    return $grouped;
  }

  /**
   * Get highest severity from enabled modules.
   *
   * @param array $modules
   *   Array of modules with severity levels.
   *
   * @return \Drupal\drupalfit\Enum\FitWeight
   *   The highest severity weight.
   */
  private function getHighestSeverity(array $modules): FitWeight {
    $severities = array_values($modules);

    return match (TRUE) {
      in_array('High', $severities, TRUE) => FitWeight::High,
      in_array('Medium', $severities, TRUE) => FitWeight::Medium,
      in_array('Low', $severities, TRUE) => FitWeight::Low,
      default => FitWeight::Info,
    };
  }

}
