<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'js_aggregation_check',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('JS Aggregation'),
  description: new TranslatableMarkup('Checks if JavaScript aggregation is enabled for better performance.'),
  successMessage: new TranslatableMarkup('JS aggregation is enabled.'),
  failureMessage: new TranslatableMarkup('JS aggregation is disabled.'),
)]
class JsAggregation extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $jsPreprocess = $this->configFactory
      ->get('system.performance')
      ->get('js.preprocess');

    if (!$jsPreprocess) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Performance settings'),
          '#url' => Url::fromRoute('system.performance_settings'),
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
