<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'opcache_enable',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('OPcache Status'),
  description: new TranslatableMarkup('Checks if PHP OPcache is enabled.'),
  successMessage: new TranslatableMarkup('OPcache is enabled.'),
  failureMessage: new TranslatableMarkup('OPcache is not enabled.'),
)]
class OpCacheEnable extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    if (!function_exists('opcache_get_status')) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}',
          '#context' => [
            'text' => $this->t('OPcache extension is not installed. Install and enable OPcache in php.ini for better performance.'),
          ],
        ]);
      return $result;
    }

    $status = opcache_get_status();
    if (empty($status) || !$status['opcache_enabled']) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}',
          '#context' => [
            'text' => $this->t('OPcache is installed but not enabled. Enable opcache.enable=1 in php.ini.'),
          ],
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
