<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\DrupalKernelInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'trusted_host_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('Trusted Host Patterns'),
  description: new TranslatableMarkup('Validates trusted host configuration.'),
  successMessage: new TranslatableMarkup('Trusted hosts configured.'),
  failureMessage: new TranslatableMarkup('Trusted hosts not configured.'),
)]
class TrustedHoldPattern extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly DrupalKernelInterface $kernel,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
    $this->getPluginId(),
    $this->label(),
    $this->fitGroup(),
    FitWeight::Ok
    );

    $settingsPhp = DRUPAL_ROOT . '/' . $this->kernel->getSitePath() . '/settings.php';

    if (!file_exists($settingsPhp)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('Create settings.php file in sites/default/.'));
      return $result;
    }

    if (!empty(Settings::get('trusted_host_patterns'))) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('Add $settings["trusted_host_patterns"] to settings.php to prevent HTTP Host header attacks.'));
    }
    return $result;
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('kernel'),
    );
  }

}
