<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\drupalfit\AuditScores;
use Drupal\drupalfit\DrupalFitConstants;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Client for DrupalFit API.
 */
class DrupalFitApiClient {
  private const int TIMEOUT = 5;

  private ?AuditScores $cachedScores = NULL;

  /**
   * Constructs DrupalFitApiClient.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Gets audit scores for the configured domain.
   */
  public function getAuditScores(): ?AuditScores {
    if ($this->cachedScores !== NULL) {
      return $this->cachedScores;
    }
    $config = $this->getConfig();
    if (!$config) {
      return NULL;
    }

    try {
      $response = $this->httpClient->request('POST', DrupalFitConstants::DRUPALFIT_API_URL . '/api-key/scores', [
        'json' => ['apiKey' => $config['api_key']],
        'headers' => ['Content-Type' => 'application/json'],
        'timeout' => self::TIMEOUT,
      ]);

      if ($response->getStatusCode() !== 200) {
        return NULL;
      }

      $data = json_decode($response->getBody()->getContents(), TRUE, 512, JSON_THROW_ON_ERROR);
      $scores = $data['data'] ?? NULL;
      $this->cachedScores = $scores ? AuditScores::fromArray($scores) : NULL;
      return $this->cachedScores;
    }
    catch (GuzzleException | \JsonException | \Exception $e) {

      return NULL;
    }
  }

  /**
   * Gets configuration.
   */
  private function getConfig(): ?array {
    $config = $this->configFactory->get('drupalfit.settings');
    $apiKey = $config->get('api_key');
    if (empty($apiKey)) {
      return NULL;
    }
    return ['api_key' => $apiKey];
  }

}
