/**
 * @file
 * DrupalFit iframe integration with postMessage API.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  let messageHandler = null;

  Drupal.behaviors.drupalFitIframe = {
    attach: function (context) {
      const iframe = context.querySelector('.drupalfit-embed-iframe');
      if (!iframe || iframe.dataset.initialized) {
        console.log('DrupalFit: No iframe or already initialized');
        return;
      }

      const apiKey = drupalSettings.drupalfit?.apiKey;
      const domain = drupalSettings.drupalfit?.domain;
      if (!apiKey) {
        console.error('DrupalFit: No API key configured');
        return;
      }

      // Always force iframe reload to ensure fresh content
      const currentCredentials = apiKey + '|' + domain;
      const timestamp = Date.now();
      const baseUrl = iframe.src.split('?')[0];
      iframe.src = baseUrl + '?t=' + timestamp;
      console.log('DrupalFit: Forcing iframe reload with URL:', iframe.src);

      iframe.dataset.credentials = currentCredentials;
      iframe.dataset.initialized = 'true';
      const allowedOrigin = new URL(iframe.src).origin;

      if (messageHandler) {
        window.removeEventListener('message', messageHandler);
      }

      messageHandler = function (event) {
        if (event.origin !== allowedOrigin) {
          return;
        }

        const data = event.data;
        if (!data || typeof data !== 'object' || data.source !== 'drupal-fit-portal') {
          return;
        }

        if (data.type === 'IFRAME_READY' || data.type === 'REQUEST_API_KEY') {
          iframe.contentWindow.postMessage(
            {
              type: 'API_KEY',
              apiKey: apiKey,
              origin: domain,
              source: 'drupal-fit-parent'
            },
            allowedOrigin
          );
        }
      };

      window.addEventListener('message', messageHandler);
    }
  };

})(Drupal, drupalSettings);
