<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Attribute;

use Drupal\Component\Plugin\Attribute\AttributeBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The fit_check attribute.
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
final class FitCheck extends AttributeBase {

  /**
   * Constructs a new FitCheck instance.
   *
   * @param string $id
   *   The plugin ID (use snake_case: core_update_check).
   * @param string $fitGroup
   *   Fit group ID (authentication, configuration, permission, system, update).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   The human-readable name (short, 2-4 words).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $description
   *   (optional) Brief description of what this check does.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $successMessage
   *   (optional) Message when check passes (affirmative statement).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $failureMessage
   *   (optional) Message when check fails (brief problem description).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $warningMessage
   *   (optional) Warning message (rarely used).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $errorMessage
   *   (optional) Error message (rarely used).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $infoMessage
   *   (optional) Informational message (rarely used).
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup[] $helpMessage
   *   (optional) Array of help messages (use setHelpMessage() in execute()).
   * @param class-string|null $deriver
   *   (optional) The deriver class.
   */
  public function __construct(
    string $id,
    public readonly string $fitGroup,
    public readonly TranslatableMarkup $label,
    public readonly ?TranslatableMarkup $description = NULL,
    public readonly ?TranslatableMarkup $successMessage = NULL,
    public readonly ?TranslatableMarkup $failureMessage = NULL,
    public readonly ?TranslatableMarkup $warningMessage = NULL,
    public readonly ?TranslatableMarkup $errorMessage = NULL,
    public readonly ?TranslatableMarkup $infoMessage = NULL,
    public readonly array $helpMessage = [],
    public readonly ?string $deriver = NULL,
  ) {
    if (!$this->validHelpMessage($helpMessage)) {
      throw new \InvalidArgumentException(
        "Help message array must contain only TranslatableMarkup objects"
      );
    }

    parent::__construct($id);

  }

  /**
   * Validates that all help messages are TranslatableMarkup objects.
   */
  public function validHelpMessage(array $helpMessage): bool {
    if (empty($helpMessage)) {
      return TRUE;
    }
    foreach ($helpMessage as $message) {
      if (!$message instanceof TranslatableMarkup) {
        return FALSE;
      }
    }
    return TRUE;
  }

}
