<?php

namespace Drupal\drupalfit\Enum;

/**
 * Severity levels for Drupal Fit checks.
 *
 * Flexible severity system for security, performance, SEO, and other audits.
 * - Negative values = issues (deduct points)
 * - Positive values = good practices (add bonus points)
 */
enum FitWeight: int {

  // Critical - Immediate action required.
  // Penalty: 25 points.
  case Critical = -10;

  // High - Significant issue.
  // Penalty: 15 points.
  case High = -5;

  // Medium - Moderate issue.
  // Penalty: 8 points.
  case Medium = -2;

  // Low - Minor issue.
  // Penalty: 3 points.
  case Low = -1;

  // Ok - Check passed, no issues found.
  // Bonus: 2 points.
  case Ok = 2;

  // Info - Informational only (no score impact).
  // Penalty: 0 points.
  case Info = 0;

  /**
   * Get penalty/bonus points for this severity level.
   */
  public function getPenalty(): int {
    return match($this) {
      self::Critical => 25,
      self::High => 15,
      self::Medium => 8,
      self::Low => 3,
      self::Ok => -2,
      self::Info => 0,
    };
  }

  /**
   * Get human-readable label.
   */
  public function getLabel(): string {
    return match($this) {
      self::Critical => 'Critical',
      self::High => 'High',
      self::Medium => 'Medium',
      self::Low => 'Low',
      self::Ok => 'Ok',
      self::Info => 'Info',
    };
  }

  /**
   * Check if this is an issue (not Ok or Info).
   */
  public function isIssue(): bool {
    return !in_array($this, [self::Ok, self::Info], TRUE);
  }

  /**
   * Get CSS class for styling.
   */
  public function getCssClass(): string {
    return match($this) {
      self::Critical => 'critical',
      self::High => 'high',
      self::Medium => 'medium',
      self::Low => 'low',
      self::Ok => 'ok',
      self::Info => 'info',
    };
  }

}
