<?php

declare(strict_types=1);

namespace Drupal\drupalfit;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Interface for fit_check plugins.
 *
 * Defines methods that must be implemented by fit_check plugins to provide
 * functionality for checking and validating fit-related operations.
 */
interface FitCheckInterface {

  /**
   * Returns the translated plugin label.
   *
   * @return string
   *   The translated label for this plugin.
   */
  public function label(): string;

  /**
   * Returns the fit group this check belongs to.
   *
   * @return string
   *   The fit group identifier.
   */
  public function fitGroup(): string;

  /**
   * Returns the description of what this check does.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable description text.
   */
  public function description(): ?TranslatableMarkup;

  /**
   * Returns the message to display when check passes successfully.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable success message.
   */
  public function successMessage(): ?TranslatableMarkup;

  /**
   * Returns the message to display when check fails.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable failure message.
   */
  public function failureMessage(): ?TranslatableMarkup;

  /**
   * Returns the warning message for this check.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable warning message.
   */
  public function warningMessage(): ?TranslatableMarkup;

  /**
   * Returns the error message for this check.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable error message.
   */
  public function errorMessage(): ?TranslatableMarkup;

  /**
   * Returns the informational message for this check.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The translatable info message.
   */
  public function infoMessage(): ?TranslatableMarkup;

  /**
   * Returns help messages for this check.
   *
   * @return array
   *   An array of help messages.
   */
  public function helpMessage(): array;

  /**
   * Performs the fit check operation.
   *
   * @return \Drupal\drupalfit\FitResult|null
   *   The result of the fit check operation, or NULL to skip rendering.
   */
  public function execute(): ?FitResult;

}
