<?php

declare(strict_types=1);

namespace Drupal\drupalfit;

/**
 * Calculates security scores from fit check results.
 *
 * Scoring system:
 * - Each group starts at 100 points
 * - Points deducted based on severity (Critical: -25, High: -15, etc.)
 * - Secure practices add bonus points (+2)
 * - Overall score is average of all group scores.
 */
class FitScoreCalculator {

  public const  MAX_SCORE = 100;

  public const  MIN_SCORE = 0;

  public function __construct(
    protected FitResultCollection $results,
    protected array $groupWeights,
    protected array $groupScoreWeights = [],
    protected array $externalProviders = [],
    protected array $externalScores = [],
  ) {}

  /**
   * Creates calculator instance.
   */
  public static function create(FitResultCollection $results, array $groupWeights = [], array $groupScoreWeights = [], array $externalProviders = [], array $externalScores = []): static {
    return new static($results, $groupWeights, $groupScoreWeights, $externalProviders, $externalScores);
  }

  /**
   * Calculates all scores.
   */
  public function calculateScores(): FitScoreResult {
    if ($this->results->isEmpty()) {
      return $this->getDefaultScores();
    }

    $groupScores = $this->calculateGroupScores();
    $overallScore = $this->calculateOverallScore($groupScores);

    return new FitScoreResult($groupScores, $overallScore);
  }

  /**
   * Gets default scores when no results.
   */
  private function getDefaultScores(): FitScoreResult {
    $groupCollection = new GroupScoreCollection();
    foreach ($this->groupWeights as $group => $weight) {
      $groupCollection->addScore(new GroupScore($group, self::MAX_SCORE, self::MAX_SCORE, $this->externalProviders[$group] ?? FALSE));
    }
    return new FitScoreResult($groupCollection, new Score(self::MAX_SCORE, self::MAX_SCORE));
  }

  /**
   * Calculates scores for all groups.
   */
  private function calculateGroupScores(): GroupScoreCollection {
    $groupCollection = new GroupScoreCollection();
    $groups = $this->getAvailableGroups();

    foreach ($groups as $group) {
      $score = $this->calculateGroupScore($group);
      $groupCollection->addScore(new GroupScore($group, $score, $score, $this->externalProviders[$group] ?? FALSE));
    }

    return $groupCollection;
  }

  /**
   * Calculates score for a specific group.
   */
  private function calculateGroupScore(string $group): int {
    if (isset($this->externalScores[$group])) {
      return $this->externalScores[$group];
    }

    $score = self::MAX_SCORE;
    $groupResults = $this->results->getByGroup($group);

    foreach ($groupResults as $result) {
      $score -= $result->weight()->getPenalty();
    }

    return max(self::MIN_SCORE, min(self::MAX_SCORE, $score));
  }

  /**
   * Calculates overall score as weighted average.
   */
  private function calculateOverallScore(GroupScoreCollection $groupCollection): Score {
    $scores = $groupCollection->getScores();

    if (empty($scores)) {
      return new Score(0, 0);
    }

    // If no score weights defined, use simple average.
    if (empty($this->groupScoreWeights)) {
      $total = 0;
      foreach ($scores as $groupScore) {
        $total += $groupScore->getScore();
      }
      $score = (int) round($total / count($scores));
      return new Score($score, $score);
    }

    // Calculate weighted average.
    $weightedTotal = 0;
    $totalWeight = 0;

    foreach ($scores as $groupScore) {
      $groupId = $groupScore->getGroup();
      $weight = $this->groupScoreWeights[$groupId] ?? 10;
      $weightedTotal += $groupScore->getScore() * $weight;
      $totalWeight += $weight;
    }

    $score = $totalWeight > 0 ? (int) round($weightedTotal / $totalWeight) : 0;
    return new Score($score, $score);
  }

  /**
   * Gets all unique groups from results.
   */
  private function getAvailableGroups(): array {
    $groups = [];
    foreach ($this->results as $result) {
      $groups[$result->group()] = TRUE;
    }
    return array_keys($groups);
  }

}
