<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\KeyValueStore\KeyValueFactory;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'core_update_check',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Drupal Core Updates'),
  description: new TranslatableMarkup('Checks for available Drupal core updates.'),
  successMessage: new TranslatableMarkup('Core is up to date.'),
  failureMessage: new TranslatableMarkup('Core updates available.'),
)]
class CoreUpdateCheck extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleHandlerInterface $moduleHandler,
    protected readonly KeyValueFactory $keyValueStore,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('keyvalue'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    if (!$this->moduleHandler->moduleExists('update')) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->t('Update module is disabled.'))
        ->setHelpMessage($this->t('Enable the Update Manager module to check for core updates.'));
    }
    else {
      $updateData = $this->keyValueStore
        ->get('update_available_releases')
        ->get('drupal');

      if ($updateData && isset($updateData['status']) && $updateData['status'] !== 1) {
        $weight = $updateData['status'] === 3 ? FitWeight::High : FitWeight::Medium;
        $version = $updateData['recommended'] ?? 'Unknown';
        $result
          ->setWeight($weight)
          ->setFailureMessage($this->failureMessage())
          ->setHelpMessage($this->t('Update to version @version to fix security vulnerabilities and bugs.', ['@version' => $version]));
      }
      else {
        $result->setSuccessMessage($this->successMessage());
      }
    }

    return $result;
  }

}
