<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Database\Connection;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'database_cred_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('Database Security'),
  description: new TranslatableMarkup('Validates database credentials strength.'),
  successMessage: new TranslatableMarkup('Database credentials are secure.'),
  failureMessage: new TranslatableMarkup('Weak database credentials detected.'),
)]
class DatabaseCredCheck extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly Connection $connection,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $databaseOption = $this->connection->getConnectionOptions();
    $username = $databaseOption['username'] ?? '';
    $password = $databaseOption['password'] ?? '';

    $weight = FitWeight::Ok;

    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      $weight
    );

    // Critical username issues.
    if (preg_match('/^(admin|root|user|test)$/i', $username)) {
      $result
        ->setWeight(FitWeight::Critical)
        ->setHelpMessage($this->t('Change username from "@username" to a unique, non-default value.', ['@username' => $username]));
    }

    // High severity password issues.
    if (strlen($password) < 8) {
      $result
        ->setWeight(FitWeight::Critical)
        ->setHelpMessage($this->t('Password is only @length characters. Use at least 16 characters with mixed case, numbers, and symbols.', ['@length' => strlen($password)]));
    }
    elseif (strlen($password) < 12) {
      $result
        ->setWeight(FitWeight::High)
        ->setHelpMessage($this->t('Increase password length to at least 16 characters for better security.'));
    }

    // Medium severity issues.
    if (strlen($username) < 8) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Use a username with at least 8 characters.'));
    }
    if (!preg_match('/[A-Z]/', $password)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Add uppercase letters to password.'));
    }
    if (!preg_match('/[a-z]/', $password)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Add lowercase letters to password.'));
    }
    if (!preg_match('/[0-9]/', $password)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Add numbers to password.'));
    }
    if (!preg_match('/[^A-Za-z0-9]/', $password)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Add special characters (!@#$%^&*) to password.'));
    }

    if ($result->weight() !== FitWeight::Ok) {
      $result->setFailureMessage($this->failureMessage());
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
