<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Template\TwigEnvironment;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'debug_mode_check',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Debug Mode'),
  description: new TranslatableMarkup('Checks if debug mode is enabled in production.'),
  successMessage: new TranslatableMarkup('Debug mode is disabled.'),
  failureMessage: new TranslatableMarkup('Debug mode is enabled.')
)]
class DebugModeCheck extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly TwigEnvironment $twigEnv,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $weight = FitWeight::Ok;

    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      $weight
    );

    // Check Twig debug.
    if ($this->twigEnv->isDebug()) {
      $result
        ->setWeight(FitWeight::High)
        ->setHelpMessage($this->t('Disable Twig debug in sites/default/services.yml (set debug: false).'));
    }

    // Check error display settings.
    $errorLevel = $this->configFactory->get('system.logging')->get('error_level');
    if ($errorLevel === 'verbose') {
      $result
        ->setWeight(FitWeight::High)
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Logging settings'),
          '#url' => Url::fromRoute('system.logging_settings'),
        ]);
    }

    // Check development services.
    $devServicesFile = DRUPAL_ROOT . '/sites/development.services.yml';
    if (file_exists($devServicesFile)) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Remove or rename development.services.yml file in production.'));
    }

    if ($result->weight() !== FitWeight::Ok) {
      $result
        ->setFailureMessage($this->failureMessage());
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('twig'),
      $container->get('config.factory'),
    );
  }

}
