<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'file_permission_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('File Permissions'),
  description: new TranslatableMarkup('Checks for writable files and directories.'),
  successMessage: new TranslatableMarkup('File permissions are secure.'),
  failureMessage: new TranslatableMarkup('Writable files detected.'),
)]
class FilePermissionCheck extends FitCheckPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    $findings = $this->findWritableFiles();
    $configFiles = $this->checkConfigFiles();
    $createStatus = $this->testFileCreation();
    $appendStatus = $this->testFileAppend();

    if (!empty($findings) || !empty($configFiles) || $createStatus || $appendStatus) {
      $issues = [];

      if (!empty($configFiles)) {
        foreach ($configFiles as $file) {
          $issues[] = $file . ' is writable';
        }
      }

      if (!empty($findings)) {
        $dirs = implode(', ', array_slice($findings, 0, 3));
        $issues[] = 'Writable directories: ' . $dirs . (count($findings) > 3 ? '...' : '');
      }

      if ($createStatus) {
        $issues[] = 'Module directory is writable';
      }

      if ($appendStatus) {
        $issues[] = 'Module files are writable';
      }

      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '<strong>{{ label }}</strong><br>{{ items|raw }}<br><br>{{ fix }}<br><a href="{{ url }}" target="_blank">{{ link }}</a>',
          '#context' => [
            'label' => $this->t('Issues found:'),
            'items' => implode('<br>', $issues),
            'fix' => $this->t('Run: chmod 444 sites/default/settings*.php sites/default/services.yml'),
            'url' => 'https://www.drupal.org/docs/administering-a-drupal-site/security-in-drupal/securing-file-permissions-and-ownership',
            'link' => $this->t('Learn more about securing file permissions'),
          ],
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

  /**
   * Checks if critical config files are writable.
   */
  private function checkConfigFiles(): array {
    require_once DRUPAL_ROOT . '/core/includes/install.inc';
    $writable = [];
    $site_path = \Drupal::getContainer()?->getParameter('site.path');

    foreach (['settings.php', 'settings.local.php', 'services.yml'] as $conf_file) {
      $full_path = $site_path . '/' . $conf_file;
      if (file_exists($full_path) && !drupal_verify_install_file($full_path, FILE_EXIST | FILE_READABLE | FILE_NOT_WRITABLE, 'file', !Settings::get('skip_permissions_hardening'))) {
        $writable[] = $full_path;
      }
    }

    return $writable;
  }

  /**
   * Finds directories that are writable by the web server.
   *
   * Checks core, modules and themes directories for write permissions
   * which could indicate a security vulnerability.
   *
   * @return array
   *   Array of writable directory paths found.
   */
  private function findWritableFiles(): array {
    $writableFiles = [];
    $checkDirs = [DRUPAL_ROOT . '/core', DRUPAL_ROOT . '/modules', DRUPAL_ROOT . '/themes'];

    foreach ($checkDirs as $dir) {
      if (is_dir($dir) && is_writable($dir)) {
        $writableFiles[] = $dir;
      }
    }

    return $writableFiles;
  }

  /**
   * Tests if files can be created in the module directory.
   *
   * Attempts to create a temporary test file in the module directory
   * to check for write permissions.
   *
   * @return bool
   *   TRUE if a file could be created, FALSE otherwise.
   */
  private function testFileCreation(): bool {
    if (!$this->moduleHandler->moduleExists('drupalfit')) {
      return FALSE;
    }

    $directory = $this->moduleHandler->getModule('drupalfit')->getPath();
    $file = DRUPAL_ROOT . '/' . $directory . '/file_write_test.' . date('Ymdhis');

    if ($fileHandle = @fopen($file, 'wb')) {
      $status = fwrite($fileHandle, date('Ymdhis') . " - Test file creation\n");
      fclose($fileHandle);
      @unlink($file);
      return (bool) $status;
    }
    return FALSE;
  }

  /**
   * Tests if existing files can be appended to.
   *
   * Creates or uses an existing IGNOREME.txt file and attempts to append
   * content to it to check for write permissions.
   *
   * @return bool
   *   TRUE if content could be appended, FALSE otherwise.
   */
  private function testFileAppend(): bool {
    if (!$this->moduleHandler->moduleExists('drupalfit')) {
      return FALSE;
    }

    $directory = $this->moduleHandler->getModule('drupalfit')->getPath();
    $file = DRUPAL_ROOT . '/' . $directory . '/IGNOREME.txt';

    if (!file_exists($file)) {
      file_put_contents($file, "Test file for permission checking\n");
    }

    if ($fileHandle = @fopen($file, 'ab')) {
      $status = fwrite($fileHandle, date('Ymdhis') . " - Test append\n");
      fclose($fileHandle);
      return (bool) $status;
    }

    return FALSE;
  }

}
