<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'last_cron_run',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('Cron Execution'),
  description: new TranslatableMarkup('Checks if cron is running regularly.'),
  successMessage: new TranslatableMarkup('Cron is running regularly.'),
  failureMessage: new TranslatableMarkup('Cron has not run recently.'),
)]
class LastCronRun extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly StateInterface $state,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('state')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $lastRun = $this->state->get('system.cron_last');

    if (!$lastRun) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('Cron has never run.'))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Cron settings'),
          '#url' => Url::fromRoute('system.cron_settings'),
        ]);
      return $result;
    }

    $now = time();
    $hoursSince = ($now - $lastRun) / 3600;

    if ($hoursSince > 48) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('Cron last ran @hours hours ago.', ['@hours' => round($hoursSince)]))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Cron settings'),
          '#url' => Url::fromRoute('system.cron_settings'),
        ]);
    }
    elseif ($hoursSince > 24) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('Cron last ran @hours hours ago.', ['@hours' => round($hoursSince)]))
        ->setHelpMessage($this->t('Configure automated cron to run more frequently.'));
    }
    elseif ($hoursSince > 12) {
      $result
        ->setWeight(FitWeight::Low)
        ->setHelpMessage($this->t('Cron last ran @hours hours ago.', ['@hours' => round($hoursSince)]))
        ->setHelpMessage($this->t('Consider running cron more frequently for better site maintenance.'));
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
