<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\ContentAndConfigGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'mismatched_entity_definitions',
  fitGroup: ContentAndConfigGroup::GROUP_ID,
  label: new TranslatableMarkup('Mismatched Entity Definitions'),
  description: new TranslatableMarkup('Checks for entity/field definition mismatches.'),
  successMessage: new TranslatableMarkup('No entity definition mismatches detected.'),
  failureMessage: new TranslatableMarkup('Entity definition mismatches detected.'),
)]
class MismatchedEntityDefinitions extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityDefinitionUpdateManagerInterface $entityDefinitionUpdateManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.definition_update_manager'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    $mismatches = [];

    try {
      $changeList = $this->entityDefinitionUpdateManager->getChangeList();
      foreach ($changeList as $entityType => $changes) {
        foreach ($changes as $changeType => $items) {
          if (!empty($items)) {
            $itemNames = is_array($items) ? implode(', ', array_keys($items)) : '';
            $mismatches[] = '<strong>' . $entityType . '</strong> - ' . str_replace('_', ' ', $changeType) . ': ' . $itemNames;
          }
        }
      }
    }
    catch (\Exception $e) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setWarningMessage($this->t('Unable to check entity definitions: @error', ['@error' => $e->getMessage()]));
      return $result;
    }

    if (empty($mismatches)) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}<br><br><strong>{{ label }}</strong><br>{{ items|raw }}<br><br><a href="{{ url }}">{{ link }}</a> {{ or }} {{ command }}',
          '#context' => [
            'text' => $this->t('Entity/field definitions are out of sync with database schema.'),
            'label' => $this->t('Mismatches found:'),
            'items' => implode('<br>', $mismatches),
            'url' => '/update.php',
            'link' => $this->t('Run updates'),
            'or' => $this->t('or run:'),
            'command' => 'drush entity:updates',
          ],
        ]);
    }

    return $result;
  }

}
