<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Checks for contributed module updates.
 */
#[FitCheck(
  id: 'php_version_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('PHP Version'),
  description: new TranslatableMarkup('Checks if PHP version is supported and secure.'),
  successMessage: new TranslatableMarkup('PHP version is supported.'),
  failureMessage: new TranslatableMarkup('PHP version issue detected.'),
)]
class PhpVersion extends FitCheckPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $currentVersion = PHP_VERSION;
    $majorMinor = PHP_MAJOR_VERSION . '.' . PHP_MINOR_VERSION;
    $drupalVersion = \Drupal::VERSION;

    $drupalSupported = $this->getDrupalSupportedPhp($drupalVersion);
    $recommended = $drupalSupported['recommended'];
    $minimum = $drupalSupported['minimum'];

    // Critical: Below Drupal minimum.
    if (version_compare(PHP_VERSION, $minimum, '<')) {
      $result
        ->setWeight(FitWeight::Critical)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('PHP @current is below Drupal @drupal minimum (@min).', [
          '@current' => $currentVersion,
          '@drupal' => $drupalVersion,
          '@min' => $minimum,
        ]))
        ->setHelpMessage($this->t('Upgrade to PHP @recommended immediately.', ['@recommended' => $recommended]));
      return $result;
    }

    // Critical: EOL versions.
    $eolVersions = ['7.4', '8.0', '8.1'];
    if (in_array($majorMinor, $eolVersions)) {
      $result
        ->setWeight(FitWeight::Critical)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage($this->t('PHP @version is end-of-life (no security updates).', ['@version' => $currentVersion]))
        ->setHelpMessage($this->t('Upgrade to PHP @recommended (recommended for Drupal @drupal).', [
          '@recommended' => $recommended,
          '@drupal' => $drupalVersion,
        ]));
      return $result;
    }

    // Low: Not on recommended version.
    if (version_compare(PHP_VERSION, $recommended, '<')) {
      $result
        ->setWeight(FitWeight::Low)
        ->setHelpMessage($this->t('PHP @current works but @recommended is recommended for Drupal @drupal.', [
          '@current' => $currentVersion,
          '@recommended' => $recommended,
          '@drupal' => $drupalVersion,
        ]));
    }

    // Check for latest patch version.
    if (PHP_MAJOR_VERSION === 8 && PHP_MINOR_VERSION === 3 && PHP_RELEASE_VERSION < 10) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setHelpMessage($this->t('Update to latest PHP 8.3.x patch for security fixes.'));
    }

    if ($result->weight() !== FitWeight::Ok) {
      $result->setFailureMessage($this->failureMessage());
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

  /**
   * Get Drupal supported PHP versions.
   */
  private function getDrupalSupportedPhp(string $drupalVersion): array {
    $major = (int) explode('.', $drupalVersion)[0];

    return match($major) {
      11 => ['minimum' => '8.3.0', 'recommended' => '8.3.0'],
      10 => ['minimum' => '8.1.0', 'recommended' => '8.3.0'],
      9 => ['minimum' => '7.4.0', 'recommended' => '8.1.0'],
      default => ['minimum' => '8.3.0', 'recommended' => '8.3.0'],
    };
  }

}
