<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'too_many_modules_enabled',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('Too Many Modules Enabled'),
  description: new TranslatableMarkup('Checks if too many modules are enabled.'),
  successMessage: new TranslatableMarkup('Module count is reasonable.'),
  failureMessage: new TranslatableMarkup('Too many modules are enabled.'),
)]
class ToManyModuleEnabled extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleExtensionList $moduleExtensionList,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('extension.list.module'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $enabledModules = array_filter($this->moduleExtensionList->getAllInstalledInfo(), fn($info) => $info['status'] ?? FALSE);
    $moduleCount = count($enabledModules);

    if ($moduleCount > 150) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->t('@count modules are enabled (>150).', ['@count' => $moduleCount]))
        ->setHelpMessage($this->t('Too many modules increase memory usage and slow down page loads.'))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Manage modules'),
          '#url' => Url::fromRoute('system.modules_list'),
        ])
        ->setHelpMessage($this->t('Consider consolidating functionality or using fewer contrib modules.'));
    }
    elseif ($moduleCount > 100) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->t('@count modules are enabled (>100).', ['@count' => $moduleCount]))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Manage modules'),
          '#url' => Url::fromRoute('system.modules_list'),
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
